# Stochastic surplus Production model in Continuous-Time (SPiCT)
#    Copyright (C) 2015-2016  Martin W. Pedersen, mawp@dtu.dk, wpsgodd@gmail.com
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


#' @name txt.stamp
#' @title Add spict version to plot
#' @param string Character string to stamp.
#' @return Nothing
#' @export
txt.stamp <- function(string = get.version(), cex=0.5) {
    #if (string == 'undef'){
    #    string <- get.version()
    #}
    if (!is.null(string)){
        if (string != '' & !is.na(string)){
            opar <- par(new = "TRUE", plt = c(0, 1, 0, 1), mfrow=c(1, 1), xpd=FALSE)
            on.exit(par(opar))
            plot(1, typ='n', xaxt='n', yaxt='n', xlab='', ylab='', bty='n') # Empty plot 
            #opar <- par(yaxt = "s", xaxt = "s")
            on.exit(par(opar))
            plt <- par("plt")
            usr <- par("usr")
            xcoord <- usr[2] + (usr[2] - usr[1])/(plt[2] - plt[1]) * (1 - plt[2]) - 0.4 * strwidth("m")
            ycoord <- usr[3] - diff(usr[3:4])/diff(plt[3:4]) * (plt[3]) + 0.4 * strheight("m")
            if (par("xlog")){
                xcoord <- 10^(xcoord)
            }
            if (par("ylog")){
                ycoord <- 10^(ycoord)
            }
            text(xcoord, ycoord, string, adj = 1, cex=cex)
        }
    }
}


#' @name warning.stamp
#' @title Add warning sign to plot
#' @return Nothing
warning.stamp <- function(){
    opar <- par(yaxt = "s", xaxt = "s", xpd = NA)
    on.exit(par(opar))
    usr <- par("usr")
    xcoord <- usr[1]
    ycoord <- usr[4] + 0.035/diff(par()$fig[3:4]) * diff(usr[3:4])
    if (par("xlog")){
        xcoord <- 10^(xcoord)
    }
    if (par("ylog")){
        ycoord <- 10^(ycoord)
    }
    points(xcoord, ycoord, pch=24, bg='yellow', col='black', cex=2, lwd=1.5)
    text(xcoord, ycoord, '!', cex=0.8)
}


#' @name add.catchunit
#' @title Add catch unit to label
#' @param lab Base label
#' @param cu Catch unit as a character string
#' @return Label with added catch unit
add.catchunit <- function(lab, cu){
    cu <- as.character(cu)
    if(cu != ''){
        out <- eval(bquote(.(lab[[1]]) *',' ~ .(cu)))
    } else {
        out <- lab
    }
    return(out)
}


#' @name arrow.line
#' @title Draw a line with arrow heads.
#' @details Add to an existing plot a continuous line with arrow heads showing the direction between each data point
#' @param x X coordinates.
#' @param y Y coordinates.
#' @param length See documentation for arrows.
#' @param angle See documentation for arrows.
#' @param code See documentation for arrows.
#' @param col See documentation for arrows.
#' @param lty See documentation for arrows.
#' @param lwd See documentation for arrows.
#' @param ... See documentation for arrows.
#' @return Nothing, but an arrow line is added to the current plot.
arrow.line <- function(x, y, length = 0.25, angle = 30, code = 2, col = par("fg"), lty = par("lty"), lwd = par("lwd"), ...){
    n <- length(x)
    for(i in 2:n) arrows(x[i-1], y[i-1], x[i], y[i], length, angle, code, col, lty, lwd, ...)
}


#' @name annual
#' @title Convert from quarterly (or other sub-annual) data to annual means or sums.
#' @param intime A time vector corresponding to the values in vec.
#' @param vec The vector of values to convert to annual means
#' @param type If type='mean' then annual mean is calculated, if type='sum' then annual sum is calculated.
#' @return A list containing the annual means and a corresponding time vector.
annual <- function(intime, vec, type='mean'){
    anntime <- intime[which(intime %% 1 ==0)]
    nanntime <- length(anntime)
    nstepvec <- rep(0, nanntime)
    floortime <- floor(intime)
    for(i in 1:nanntime) nstepvec[i] <- sum(anntime[i]==floortime)
    nsteps <- max(nstepvec)
    # Remove years that are not full
    anntime <- anntime[which(nstepvec==max(nstepvec))]
    nanntime <- length(anntime)
    annvec <- rep(0, nanntime)
    for(i in 1:nanntime){
        inds <- which(anntime[i]==floortime)
        if(type=='mean'){
            annvec[i] <- mean(vec[inds])
        }
        if(type=='sum'){
            annvec[i] <- sum(vec[inds])
        }
    }
    return(list(anntime=anntime, annvec=annvec))
}


#' @name refpointci
#' @title Draw CI around a reference point using polygon
#' @param t Time vector.
#' @param ll Lower limit.
#' @param ul Upper limit.
#' @param cicol Colour of polygon
#' @return Spline design matrix.
refpointci <- function(t, ll, ul, cicol='ivory2'){
    tplus <- c(t[1]*0.8, t, tail(t, 1)*1.2)
    llplus <- c(ll[1], ll, tail(ll, 1))
    ulplus <- c(ul[1], ul, tail(ul, 1))
    polygon(c(tplus, rev(tplus)), c(llplus, rev(ulplus)), col=cicol, border=cicol)
}


#' @name make.ellipse
#' @title Calculate confidence ellipsis.
#' @details Calculates the confidence ellipsis of two reported model parameters. This is particularly useful as a detailed view of the uncertainty of two correlated parameters.
#' @param inds Indices of the two reported model parameters.
#' @param rep A result report as generated by running fit.spict.
#' @return A matrix with two columns containing the x and y coordinates of the ellipsis.
make.ellipse <- function(inds, rep){
    #require(ellipse)
    covBF <- rep$cov[inds,inds]
    corBF <- cov2cor(covBF)
    parBF <- rep$value[inds]
    return(ellipse::ellipse(corBF[1,2], scale=sqrt(diag(covBF)), centre=parBF, npoints=300))
}


#' @name season.cols
#' @title Load season colors.
#' @param modin Time vector modulo 1.
#' @return Vector containing season colors.
season.cols <- function(modin){
    # Make colors
    seasons <- colorRamp(c('blue', 'green', 'gold', 'red', 'blue'))
    ncols <- 13
    cols <- seasons(seq(0, 1, length=ncols))
    rgbcols <- rgb(cols[, 1]/255, cols[, 2]/255, cols[, 3]/255) # Color no 13 = color no 1, only use 1:12
    # Find colors corresponding to modin
    breaks <- seq(0, 1, length=ncols) # Monthly breaks
    ind <- cut(modin, breaks, right=FALSE, labels=FALSE)
    ind[is.na(ind)] <- 1
    return(rgbcols[ind])
}


#' @name true.col
#' @title Load color of true values from simulation.
#' @return Color vector
true.col <- function() rgb(1, 165/255, 0, alpha=0.7) # 'orange'


#' @name plot.col
#' @title Plot model points colored depending on the quarter to which they belong.
#' @param time Time vector.
#' @param obs Observation vector (or residual vector).
#' @param obsx Second observation vector for use as independent variable instead of time.
#' @param pch Point character.
#' @param add If TRUE plot is added to the current plot.
#' @param typ Plot type.
#' @param do.line If TRUE draw a line between points.
#' @param add.legend If TRUE add legend containing information on quarters.
#' @param ... Additional plotting arguments.
#' @return Nothing.
plot.col <- function(time, obs, obsx=NULL, pch=1, add=FALSE, typ='p', do.line=TRUE, add.legend=FALSE, ...){
    if(is.null(obsx)){
        x <- time
    } else {
        x <- obsx
    }
    nobs <- length(obs)
    mods <- time%%1
    cols <- season.cols(mods)
    if(!add) plot(x, obs, typ='n', ...)
    if(typ=='p' & do.line) lines(x, obs, col='lightgray')
    if(typ=='p') points(x, obs, col=1, pch=20+pch, bg=cols)
    if(typ=='l'){
        qs <- unique(mods)
        qcols <- season.cols(qs)
        nintv <- length(qs)
        if(nintv <= 4){
            for(i in 1:nintv){
                inds <- which(qs[i] == mods)
                lines(x[inds], obs[inds], col=qcols[i], lty=1)
            }
        } else {
            lines(x, obs, col='lightgray')
            warning('More than four observed sub-annual time points (', nintv, '), plotting only one line.')
        }
    }
    if(add.legend) add.col.legend()
    if(!add) box(lwd=1.5)
}


#' @name add.col.legend.hor
#' @title Add a legend explaining colors of points (horizontal orientation)
#' @return Nothing.
add.col.legend.hor <- function(){
    pusr <- par('usr')
    mods <- seq(0, 1, length=13)
    rgbcols <- season.cols(mods)
    nbar <- length(rgbcols)-1
    barwidth <- diff(pusr[1:2])
    dbw <- barwidth/nbar
    pxmax <- pusr[2]
    pymax <- pusr[4]
    barheight <- 0.05*diff(pusr[3:4])
    barx <- pxmax-barwidth
    bary <- pymax-barheight
    for(i in 1:nbar){
        xst <- barx+(i-1)*dbw
        rect(xst,bary,xst+dbw,pymax,col=rgbcols[i],lty=0)
    }
    rect(barx,bary,pxmax,pymax)
    labx <- seq(barx, barx+barwidth, length=5)
    laby <- pusr[4]-barheight
    text(labx[1:4]+dbw/2, laby, c('Jan', 'Apr', 'Jul', 'Oct'), pos=1)
}


#' @name add.col.legend
#' @title Add a legend explaining colors of points (vertical orientation)
#' @return Nothing.
add.col.legend <- function(){
    rel <- (par()$fin/par()$pin)[2]
    yfac <- 3*rel - 3
    opar <- par(xpd=TRUE)
    on.exit(par(opar))
    pusr <- par('usr')
    mods <- seq(0, 1, length=13)
    rgbcols <- season.cols(mods)
    nbar <- length(rgbcols)-1
    barwidth <- 0.05*diff(pusr[1:2])
    barheight <- 0.1*yfac*diff(pusr[3:4])
    dbh <- barheight/nbar
    pxmax <- pusr[2]
    pymax <- pusr[4]
    barx <- pxmax-barwidth
    bary <- pymax+barheight
    for(i in 1:nbar){
        #xst <- barx+(i-1)*dbw
        yst <- bary - (i-1)*dbh
        rect(barx, yst, pxmax, yst+dbh, col=rgbcols[i], lty=0)
    }
    #rect(barx, bary, pxmax, pymax)
    laby <- seq(bary, pymax, length=5)
    labx <- pxmax
    text(labx, laby[1:4], c('Jan', 'Apr', 'Jul', 'Oct'), pos=4, cex=0.8)
}


#' @name plotspict.biomass
#' @title Plot estimated biomass.
#' @details Plots estimated biomass, Bmsy with confidence limits.
#' @param rep A result report as generated by running fit.spict.
#' @param logax Take log of y-axis? default: FALSE
#' @param main Title of plot.
#' @param ylim Limits for y-axis.
#' @param plot.obs If TRUE observations are plotted.
#' @param qlegend If TRUE legend explaining colours of observation data is plotted.
#' @param xlab Label of x-axis.
#' @param ylab Label of y-axis.
#' @param rel.axes Plot secondary y-axis contatning relative level of F.
#' @param rel.ci Plot confidence interval for relative level of F.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.biomass(rep)
#' @export
plotspict.biomass <- function(rep, logax=FALSE, main='Absolute biomass', ylim=NULL,
                              plot.obs=TRUE, qlegend=TRUE, xlab='Time', ylab=NULL,
                              rel.axes=TRUE, rel.ci=TRUE, stamp=get.version()){
    if(!'sderr' %in% names(rep)){
        ylabflag <- is.null(ylab)
        mar <- c(5.1, 4.3, 4.1, 4.1)
        if(dev.cur()==1){ # If plot is not open
            opar <- par(mar=mar)
            on.exit(par(opar))
        }
        if(dev.cur()==2){ # If plot is open, check if it is a 1x1 plot
            if(all(par()$mfrow == c(1, 1))){
                opar <- par(mar=mar)
                on.exit(par(opar))
            }
        }
        log <- ifelse(logax, 'y', '')
        inp <- rep$inp
        # Biomass plot
        Best <- get.par('logB', rep, exp=TRUE)
        ns <- dim(Best)[1]
        Kest <- get.par('logK', rep, exp=TRUE)
        Bmsy <- get.par('logBmsy', rep, exp=TRUE)
        Bmsyvec <- get.msyvec(inp, Bmsy)
        qest <- get.par('logq', rep, exp=TRUE)
        BB <- get.par('logBBmsy', rep, exp=TRUE)
        Bp <- get.par('logBp', rep, exp=TRUE)
        scal <- 1
        cicol <- 'lightgray'
        if (inp$nindex==0){
            nindexseq <- numeric(0)
            obsI <- NULL
        } else {
            nindexseq <- 1:inp$nindex
            obsI <- list()
        }
        for(i in nindexseq) obsI[[i]] <- inp$obsI[[i]]/qest[inp$mapq[i], 2]
        fininds <- which(Best[, 5] < 5) # Use CV to check for large uncertainties
        BBfininds <- unname(which(is.finite(BB[, 1]) & is.finite(BB[, 3]))) # Use CV to check for large uncertainties
        if(length(ylim)!=2) ylim <- range(BB[BBfininds, 1:3]/scal*Bmsy[2], Best[fininds, 1:3], Bp[2], unlist(obsI), 0.95*Bmsy[1], 1.05*Bmsy[3], na.rm=TRUE)/scal
        ylim[2] <- min(c(ylim[2], 3*max(Best[fininds, 2], unlist(obsI)))) # Limit upper limit
        #if(main==-1) main <- 'Absolute biomass'
        if(ylabflag){
            ylab <- expression(B[t])
            ylab <- add.catchunit(ylab, inp$catchunit)
        }
        plot(inp$time, Best[,2]/scal, typ='n', xlab=xlab, ylab=ylab, main=main, ylim=ylim, xlim=range(c(inp$time, tail(inp$time, 1) + 0.5)), log=log)
        if(rel.axes){
            axis(4, labels=pretty(ylim/Bmsy[2]), at=pretty(ylim/Bmsy[2])*Bmsy[2])
            mtext(expression(B[t]/B[MSY]), side=4, las=0, line=2.2, cex=par('cex'))
        }
        if (all(is.finite(unlist(Bmsyvec)))){
            polygon(c(inp$time, rev(inp$time)), c(Bmsyvec$ll,rev(Bmsyvec$ul)),
                    col=cicol, border=cicol)
        }
        cicol2 <- rgb(0, 0, 1, 0.1)
        if (!'yearsepgrowth' %in% names(inp) & rel.ci){
            polygon(c(inp$time[BBfininds], rev(inp$time[BBfininds])),
                    c(BB[BBfininds, 1], rev(BB[BBfininds, 3]))/scal*Bmsy[2],
                    col=cicol2, border=cicol2)
        }
        abline(v=inp$time[inp$indlastobs], col='gray')
        if(plot.obs){
            for(i in nindexseq) plot.col(inp$timeI[[i]], inp$obsI[[i]]/qest[inp$mapq[i], 2], pch=i, do.line=FALSE, cex=0.6, add=TRUE, add.legend=FALSE)
            if(qlegend){
                subyears <- unique(unlist(inp$timeI)%%1)
                plot.col(subyears, numeric(length(subyears)), typ='n', add=TRUE, add.legend=TRUE) # Only plot legend
            }
            # Highlight influential index observations
            if('infl' %in% names(rep)){
                infl <- rep$infl$infl
                indslast <- inp$nobsC # Start after catch observations
                for(i in nindexseq){
                    iinds <- indslast + 1:inp$nobsI[i]
                    infl2 <- infl[iinds, ]
                    cols <- apply(!is.na(infl2), 1, sum)
                    ncols <- length(unique(cols))
                    inds <- which(cols>0)
                    points(inp$timeI[[i]][inds], inp$obsI[[i]][inds]/qest[inp$mapq[i], 2], pch=21, cex=0.9, bg=cols[inds])
                }
            }
        }
        if('true' %in% names(inp)){
            lines(inp$true$time, inp$true$B/scal, col=true.col()) # Plot true
            abline(h=inp$true$Bmsy, col=true.col(), lty=1)
            abline(h=inp$true$Bmsy, col='black', lty=3)
        }
        lines(inp$time[inp$indest], Best[inp$indest,2]/scal, col='blue', lwd=1.5)
        lines(inp$time[inp$indpred], Best[inp$indpred,2]/scal, col='blue', lty=3)
        lines(inp$time, Bmsyvec$msy, col='black')
        # B CI
        #if(inp$phases$logq>0){
        lines(inp$time[inp$indest], Best[inp$indest,1]/scal, col=4, lty=2, lwd=1.5)
        lines(inp$time[inp$indest], Best[inp$indest,3]/scal, col=4, lty=2, lwd=1.5)
        lines(inp$time[inp$indpred], Best[inp$indpred,1]/scal, col=4, lty=2)
        lines(inp$time[inp$indpred], Best[inp$indpred,3]/scal, col=4, lty=2)
        #}
        # B/Bmsy CI
        cicol3 <- rgb(0, 0, 1, 0.2)
        if(!'yearsepgrowth' %in% names(inp)){
            lines(inp$time[inp$indest], BB[inp$indest,1]/scal*Bmsy[2], col=cicol3, lty=1, lwd=1)
            lines(inp$time[inp$indest], BB[inp$indest,3]/scal*Bmsy[2], col=cicol3, lty=1, lwd=1)
            lines(inp$time[inp$indpred], BB[inp$indpred,1]/scal*Bmsy[2], col=cicol3, lty=1, lwd=1)
            lines(inp$time[inp$indpred], BB[inp$indpred,3]/scal*Bmsy[2], col=cicol3, lty=1, lwd=1)
        }
        if('yearsepgrowth' %in% names(inp)) abline(v=inp$yearsepgrowth, col=3)
        box(lwd=1.5)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.bbmsy
#' @title Plot estimated B/Bmsy.
#' @details Plots estimated B/Bmsy.
#' @param rep A result report as generated by running fit.spict.
#' @param logax Take log of y-axis? default: FALSE
#' @param main Title of plot.
#' @param ylim Limits for y-axis.
#' @param plot.obs If TRUE observations are plotted.
#' @param qlegend If TRUE legend explaining colours of observation data is plotted.
#' @param lineat Draw horizontal line at this y-value.
#' @param xlab Label of x-axis.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.bbmsy(rep)
#' @export
plotspict.bbmsy <- function(rep, logax=FALSE, main='Relative biomass', ylim=NULL,
                            plot.obs=TRUE, qlegend=TRUE, lineat=1, xlab='Time',
                            stamp=get.version()){
    if(!'sderr' %in% names(rep)){
        log <- ifelse(logax, 'y', '')
        inp <- rep$inp
        # Biomass plot
        Kest <- get.par('logK', rep, exp=TRUE, fixed=TRUE)
        Bmsy <- get.par('logBmsy', rep, exp=TRUE)
        Bmsyvec <- get.msyvec(inp, Bmsy)
		if (!all(is.na(Bmsyvec$msy))){ # Don't plot if all are NA
        qest <- get.par('logq', rep, fixed=TRUE, exp=TRUE)
        BB <- get.par('logBBmsy', rep, exp=TRUE)
        ns <- dim(BB)[1]
        cicol <- 'lightgray'
        if (inp$nindex == 0){
            nindexseq <- numeric(0)
            obsI <- NULL
        } else {
            nindexseq <- 1:inp$nindex
            obsI <- list()
            for(i in nindexseq) obsI[[i]] <- inp$obsI[[i]]/qest[inp$mapq[i], 2]/Bmsy[1,2]
        }
        fininds <- which(apply(BB, 1, function(x) all(is.finite(x))))
        BBfininds <- which(is.finite(BB[, 1]) & is.finite(BB[, 3]))
        if(length(ylim) != 2) ylim <- range(c(lineat, BB[fininds, 1:3], unlist(obsI), 1), na.rm=TRUE)
        ylim[2] <- min(c(ylim[2], 3*max(BB[fininds, 2], unlist(obsI)))) # Limit upper limit
        plot(inp$time, BB[,2], typ='n', xlab=xlab, ylab=expression(B[t]/B[MSY]), ylim=ylim, xlim=range(c(inp$time, tail(inp$time, 1) + 0.5)), log=log, main=main)
        cicol2 <- rgb(0, 0, 1, 0.1)
        polygon(c(inp$time[BBfininds], rev(inp$time[BBfininds])), c(BB[BBfininds, 1], rev(BB[BBfininds, 3])), col=cicol2, border=cicol2)
        abline(v=inp$time[inp$indlastobs], col='gray')
        if(plot.obs){
            for(i in nindexseq) plot.col(inp$timeI[[i]], obsI[[i]], pch=i, do.line=FALSE, cex=0.6, add=TRUE, add.legend=FALSE)
            if(qlegend){
                subyears <- unique(unlist(inp$timeI)%%1)
                plot.col(subyears, numeric(length(subyears)), typ='n', add=TRUE, add.legend=TRUE) # Only plot legend
            }
            # Highlight influential index observations
            if('infl' %in% names(rep)){
                infl <- rep$infl$infl
                indslast <- inp$nobsC # Start after catch observations
                for(i in nindexseq){
                    iinds <- indslast + 1:inp$nobsI[i]
                    infl2 <- infl[iinds, ]
                    cols <- apply(!is.na(infl2), 1, sum)
                    ncols <- length(unique(cols))
                    inds <- which(cols>0)
                    points(inp$timeI[[i]][inds], inp$obsI[[i]][inds]/qest[inp$mapq[i], 2]/Bmsy[2], pch=21, cex=0.9, bg=cols[inds])
                }
            }
        }
        if('true' %in% names(inp)){
            lines(inp$true$time, inp$true$B/inp$true$Bmsy, col=true.col()) # Plot true
        }
        lines(inp$time[inp$indest], BB[inp$indest,2], col='blue', lwd=1.5)
        lines(inp$time[inp$indpred], BB[inp$indpred,2], col='blue', lty=3)
        cicol3 <- rgb(0, 0, 1, 0.2)
        lines(inp$time[inp$indest], BB[inp$indest,1], col=cicol3, lty=1, lwd=1)
        lines(inp$time[inp$indest], BB[inp$indest,3], col=cicol3, lty=1, lwd=1)
        lines(inp$time[inp$indpred], BB[inp$indpred,1], col=cicol3, lty=1, lwd=1)
        lines(inp$time[inp$indpred], BB[inp$indpred,3], col=cicol3, lty=1, lwd=1)
        abline(h=lineat)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        box(lwd=1.5)
        } else {
            plot(1, typ='n', xlab='', ylab='', xaxt='n', yaxt='n', main=paste('Bmsy=NA!', main))
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.osar
#' @title Plot one-step-ahead residuals
#' @details Plots observed versus predicted catches.
#' @param rep A result report as generated by running fit.spict.
#' @param collapse.I Collapse index residuals into one plot. Default: TRUE.
#' @param qlegend Plot legend for quarters.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' rep <- calc.osa.resid(rep)
#' plotspict.osar(rep)
#' @export
plotspict.osar <- function(rep, collapse.I=TRUE, qlegend=TRUE){
    if('osar' %in% names(rep)){
        inp <- rep$inp
        Cscal <- 1
        Cpred <- rep$osar$logCpred
        fun <- function(time, res, add=FALSE, add.legend=FALSE, col=1, pch=1, ...){
            nrem <- length(time) - length(res)
            if(nrem>0) time <- time[-nrem]
            plot.col(time, res, pch=pch, add=add, add.legend=add.legend, typ='p', xlab='Time', ...)
            dum <- rep(NA, length(res))
            dum[is.na(res)] <- 0
            text(time, dum, labels='NA', cex=0.8, col=col)
        }
        # Catches
        pval <- round(as.list(rep$diagn)$biasC.p, 4)
        colmain <- ifelse(pval < 0.05, 'red', 'forestgreen')
        fun(rep$osar$timeC, rep$osar$logCpres, add.legend=qlegend, ylab='Catch OSA residuals',
            main=paste0('Bias p-val: ', pval), col.main=colmain)
        abline(h=0, lty=3)
        # Effort
        if (inp$nobsE > 0){
            pval <- round(as.list(rep$diagn)$biasE.p, 4)
            colmain <- ifelse(pval < 0.05, 'red', 'forestgreen')
            fun(rep$osar$timeE, rep$osar$logEpres, add.legend=qlegend, ylab='Effort OSA residuals',
                main=paste0('Bias p-val: ', pval), col.main=colmain)
            abline(h=0, lty=3)
        }
        # Indices
        if (inp$nindex > 0){
            pval <- round(as.list(rep$diagn)$biasI1.p, 4)
            colmain <- ifelse(pval < 0.05, 'red', 'forestgreen')
            if (collapse.I){
                ylim <- range(unlist(rep$osar$logIpres), na.rm=TRUE)
                xlim <- range(unlist(rep$osar$timeI))
            } else {
                ylim <- range(rep$osar$logIpres[[1]], na.rm=TRUE)
                xlim <- range(rep$osar$timeI[[1]])
            }
            ylab <- ifelse(collapse.I, 'Index OSA residuals', 'Index 1 OSA residuals')
            main <- paste0(ifelse(collapse.I, 'Index 1 ', ''), 'Bias p-val: ', pval)
            fun(rep$osar$timeI[[1]], rep$osar$logIpres[[1]], ylab=ylab,
                col=1, xlim=xlim, ylim=ylim, main=main, col.main=colmain)
            abline(h=0, lty=3)
            if(rep$inp$nindex>1){
                for(i in 2:rep$inp$nindex){
                    ylim <- range(rep$osar$logIpres[[i]], na.rm=TRUE)
                    xlim <- range(unlist(rep$osar$timeI[[i]]))
                    if(!collapse.I){
                        pval <- round(as.list(rep$diagn)[[paste0('biasI', i, '.p')]], 4)
                        #pval <- round(rep$osar$logIpbias[[i]]$p.value, 4)
                        colmain <- ifelse(pval<0.05, 'red', 'forestgreen')
                        #main <- paste0('I', i, ' bias p-val: ', pval)
                        main <- paste0('Bias p-val: ', pval)
                    } else {
                        main <- ''
                    }
                    fun(rep$osar$timeI[[i]], rep$osar$logIpres[[i]], add=collapse.I,
                        ylab=paste('Index', i, 'OSA residuals'),
                        col=1, pch=i, xlim=xlim, ylim=ylim, main=main, col.main=colmain)
                    if(!collapse.I){
                        abline(h=0, lty=3)
                    }
                }
            }
        }
    } else {
        stop('Could not find "osar" key in rep list! did you run calc.osa.resid?')
    }
}


#' @name plotspict.diagnostic
#' @title Plot model diagnostic (data, residuals, and more)
#' @param rep A result report as generated by running fit.spict.
#' @param lag.max Maximum lag to use in acf calculations.
#' @param qlegend If TRUE plot a legend showing quarter of year information.
#' @param plot.data If TRUE plot data in the top row (this option is only applied if osa residuals have been calculated).
#' @param mfcol If TRUE plot plots columnwise (FALSE => rowwise).
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' rep <- calc.osa.resid(rep)
#' plotspict.diagnostic(rep)
#' @export
plotspict.diagnostic <- function(rep, lag.max=4, qlegend=TRUE, plot.data=TRUE, mfcol=FALSE,
                                 stamp=get.version()){
    repflag <- FALSE
    mar <- c(4.7, 4.1, 2.5, 2)
    #op <- par()
    if('obsC' %in% names(rep)){ # rep is an input list
        inp <- check.inp(rep)
        if(inp$nindex==1) mfrow <- c(2, 1)
        if(inp$nindex==2) mfrow <- c(3, 1)
        if(inp$nindex==3) mfrow <- c(2, 2)
        if(inp$nindex %in% 4:5) mfrow <- c(3, 2)
        if(inp$nindex > 5) mfrow <- c(4, 4)
    }
    if('inp' %in% names(rep)){ # rep is a results list
        repflag <- TRUE
        inp <- rep$inp
        if(inp$nindex %in% 1:2) mfrow <- c(2, 2)
        if(inp$nindex %in% 3:4) mfrow <- c(3, 2)
        if(inp$nindex %in% 5:7) mfrow <- c(3, 3)
    }
    # Determine number of plots
    if('osar' %in% names(rep)){
        mfrow <- c(3 + as.numeric(plot.data), inp$nindex + 1 + as.numeric(rep$inp$nobsE > 0))
    } else {
        #cat('No OSAR found in input, run calc.osa.resid to get all diagnostics.\n')
    }
    if(mfcol){
        opar <- par(mfcol=rev(mfrow), mar=mar)
    } else {
        opar <- par(mfrow=mfrow, mar=mar)
    }
    on.exit(par(opar))
    
    # Plot data
    if(plot.data){
        plot.col(inp$timeC, log(inp$obsC), ylab='log catch data', main='Catch', xlab='Time')
        if (inp$nobsE > 0){
            plot.col(inp$timeE, log(inp$obsE), ylab='log effort data', main='Effort', xlab='Time')
        }
        if (inp$nindex > 0){
            for(i in 1:inp$nindex)
                plot.col(inp$timeI[[i]], log(inp$obsI[[i]]), ylab=paste('log index',i,'data'),
                         main=paste('Index', i), pch=i, xlab='Time')
        }
    }
   
    # OSAR plots
    osar.acf.plot <- function(res, lag.max, pval, ylab){
        inds <- which(acf.signf(res, lag.max=lag.max))
        if (length(inds) > 0){
            txt <- paste0('lag.signf: ', paste0(inds, collapse=','))
        } else {
            txt <- ''
        }
        colmain <- ifelse(pval < 0.05, 'red', 'forestgreen')
        acf(res, main='', lag.max=lag.max, ylab=ylab)
        title(main=paste0('LBox p-val: ', pval), col.main=colmain)
        legend('topright', legend=NA, title=txt, col=2, bty='n', pt.cex=0, text.col=2)
        box(lwd=1.5)
    }
    osar.qq.plot <- function(res, pval){
        colmain <- ifelse(pval < 0.05, 'red', 'forestgreen')
        qqnorm(res, main=paste0('Shapiro p-val: ', pval), col.main=colmain)
        qqline(res)
        box(lwd=1.5)
    }
    if('osar' %in% names(rep)){
        plotspict.osar(rep, collapse.I=FALSE, qlegend=qlegend)
        # Catch ACF
        pvalacfC <- round(as.list(rep$diagn)$LBoxC.p, 4)
        resC <- rep$osar$logCpres[!is.na(rep$osar$logCpres)]        
        osar.acf.plot(resC, lag.max, pvalacfC, ylab='Catch ACF')
        # Effort ACF
        if (inp$nobsE > 0){
            pvalacfE <- round(as.list(rep$diagn)$LBoxE.p, 4)
            resE <- rep$osar$logEpres[!is.na(rep$osar$logEpres)]
            osar.acf.plot(resE, lag.max, pvalacfE, ylab='Effort ACF')
        }
        # Index ACF
        if (inp$nindex > 0){
            inds <- grep('LBoxI', names(rep$diagn))
            nms <- names(rep$diagn)[inds]
            nos <- as.numeric(unlist(regmatches(nms, gregexpr('[0-9]+', nms))))
            resI <- list()
            for(i in 1:inp$nindex){
                pvalacfI <- round(as.list(rep$diagn)[[nms[i]]], 4)
                resI[[i]] <- rep$osar$logIpres[[i]][!is.na(rep$osar$logIpres[[i]])]
                osar.acf.plot(resI[[nos[i]]], lag.max, pvalacfI, ylab=paste0('Index ', i, ' ACF'))
            }
        }
        # Catch QQ
        pvalC <- round(as.list(rep$diagn)$shapiroC.p, 4)
        osar.qq.plot(resC, pvalC)
        # Effort QQ
        if (inp$nobsE > 0){
            pvalE <- round(as.list(rep$diagn)$shapiroE.p, 4)
            osar.qq.plot(resE, pvalE)
        }
        # Index QQ
        if (inp$nindex > 0){
            inds <- grep('shapiroI', names(rep$diagn))
            nms <- names(rep$diagn)[inds]
            nos <- as.numeric(unlist(regmatches(nms, gregexpr('[0-9]+', nms))))
            for(i in 1:inp$nindex){
                pvalI <- round(as.list(rep$diagn)[[nms[i]]], 4)
                osar.qq.plot(resI[[nos[i]]], pvalI)
            }
        }
    }
    txt.stamp(stamp)
}


#' @name plotspict.f
#' @title Plot estimated fishing mortality.
#' @details Plots estimated fishing mortality with Fmsy and associated confidence interval.
#' @param rep A result report as generated by running fit.spict.
#' @param logax Take log of y-axis? default: FALSE
#' @param main Title of plot.
#' @param ylim Limits for y-axis.
#' @param plot.obs If TRUE observations are plotted.
#' @param qlegend If TRUE legend explaining colours of observation data is plotted.
#' @param xlab Label of x-axis.
#' @param ylab Label of y-axis.
#' @param rel.axes Plot secondary y-axis contatning relative level of F.
#' @param rel.ci Plot confidence interval for relative level of F.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.f(rep)
#' @export
plotspict.f <- function(rep, logax=FALSE, main='Absolute fishing mortality', ylim=NULL,
                        plot.obs=TRUE, qlegend=TRUE, xlab='Time', ylab=NULL, rel.axes=TRUE,
                        rel.ci=TRUE, stamp=get.version()){
    if(!'sderr' %in% names(rep)){
        ylabflag <- is.null(ylab) # If null then not manually specified
        omar <- par()$mar
        mar <- c(5.1, 4.3, 4.1, 4.1)
        if(dev.cur()==1){ # If plot is not open
            opar <- par(mar=mar)
            on.exit(par(opar))
        }
        if(dev.cur()==2){ # If plot is open, check if it is a 1x1 plot
            if(all(par()$mfrow == c(1, 1))){
                opar <- par(mar=mar)
                on.exit(par(opar))
            }
        }
        log <- ifelse(logax, 'y', '')
        inp <- rep$inp
        cicol <- 'lightgray'
        qf <- get.par('logqf', rep, exp=TRUE)
        Fest <- get.par('logFs', rep, exp=TRUE)
        logFest <- get.par('logFs', rep)
        FF <- get.par('logFFmsy', rep, exp=TRUE)
        logFF <- get.par('logFFmsy', rep)
        Fmsy <- get.par('logFmsy', rep, exp=TRUE)
        Fmsyd <- get.par('logFmsyd', rep, exp=TRUE)
        Fmsyvec <- get.msyvec(inp, Fmsy)
        if(min(inp$dtc) < 1){
            # Annual    
            al1 <- annual(inp$time, logFest[, 1])
            al2 <- annual(inp$time, logFest[, 2])
            al3 <- annual(inp$time, logFest[, 3])
            inds <- which(!is.na(al1$annvec) & al2$anntime<=tail(inp$time[inp$indest],1))
            cl <- exp(al1$annvec[inds])
            cu <- exp(al3$annvec[inds])
            inds <- which(!is.na(al2$annvec) & al2$anntime<=tail(inp$time[inp$indest],1))
            time <- al1$anntime[inds]
            F <- exp(al2$annvec[inds])
            inds <- which(!is.na(al1$annvec) & al2$anntime>=tail(inp$time[inp$indest],1))
            clp <- exp(al1$annvec[inds])
            cup <- exp(al3$annvec[inds])
            inds <- which(!is.na(al2$annvec) & al2$anntime>=tail(inp$time[inp$indest],1))
            timep <- al1$anntime[inds]
            Fp <- exp(al2$annvec[inds])
            al1f <- annual(inp$time, logFF[, 1])
            al2f <- annual(inp$time, logFF[, 2])
            al3f <- annual(inp$time, logFF[, 3])
            inds <- which(!is.na(al1f$annvec))
            clf <- exp(al1f$annvec[inds])*Fmsy[2]
            cuf <- exp(al3f$annvec[inds])*Fmsy[2]
            inds <- which(!is.na(al2f$annvec))
            timef <- al2f$anntime[inds]
            Ff <- exp(al2f$annvec[inds])*Fmsy[2]
        } else {
            time <- inp$time[inp$indest]
            cl <- Fest[inp$indest, 1]
            F <- Fest[inp$indest, 2]
            cu <- Fest[inp$indest, 3]
            timep <- inp$time[inp$indpred]
            clp <- Fest[inp$indpred, 1]
            Fp <- Fest[inp$indpred, 2]
            cup <- Fest[inp$indpred, 3]
            timef <- inp$time
            clf <- FF[, 1]*Fmsy[2]
            Ff <- Fest[, 2] #*Fmsy[2]
            cuf <- FF[, 3]*Fmsy[2]
        }
        flag <- length(cu)==0 | all(!is.finite(cu))
        ylimflag <- length(ylim)!=2 # If FALSE ylim is manually specified
        if(flag){
            fininds <- which(is.finite(Ff))
            if(ylimflag) ylim <- range(c(Ff, Fmsy[1:3], tail(Fest[, 2],1)), na.rm=TRUE)
        } else {
            fininds <- which(apply(cbind(clf, cuf), 1, function(x) all(is.finite(x))))
            if(ylimflag) ylim <- range(c(cl[fininds], cu[fininds], clf[fininds], cuf[fininds], tail(Fest[, 2],1)), na.rm=TRUE)
        }
        if(ylimflag) ylim[2] <- min(c(ylim[2], 3*max(Ff[fininds]))) # Limit upper limit
        #if(main==-1) main <- 'Absolute fishing mortality'
        if(ylabflag) ylab <- expression(F[t])
        plot(timef, Ff, typ='n', main=main, ylim=ylim, col='blue', ylab=ylab, xlab=xlab, xlim=range(c(inp$time, tail(inp$time, 1) + 0.5)))
        if(rel.axes){
            axis(4, labels=pretty(ylim/Fmsy[2]), at=pretty(ylim/Fmsy[2])*Fmsy[2])
            mtext(expression(F[t]/F[MSY]), side=4, las=0, line=2.2, cex=par('cex'))
        }
        polygon(c(inp$time, rev(inp$time)), c(Fmsyvec$ll,rev(Fmsyvec$ul)), col=cicol, border=cicol)
        cicol2 <- rgb(0, 0, 1, 0.1)
        if(!flag & !'yearsepgrowth' %in% names(inp) & rel.ci) polygon(c(timef[fininds], rev(timef[fininds])), c(clf[fininds], rev(cuf[fininds])), col=cicol2, border=cicol2)
        if(min(inp$dtc) < 1){ # Plot estimated sub annual F 
            lines(inp$time, Fest[, 2], col=rgb(0, 0, 1, 0.4))
        }
        abline(v=inp$time[inp$indlastobs], col='gray')
        if (plot.obs) plot.col(inp$timeE, inp$obsE/inp$dte*qf[2], cex=0.7, do.line=FALSE, add=TRUE, add.legend=qlegend)
        if('true' %in% names(inp)){
            lines(inp$true$time, inp$true$Fs, col=true.col()) # Plot true
            abline(h=inp$true$Fmsy, col=true.col(), lty=1)
            abline(h=inp$true$Fmsy, col='black', lty=3)
        }
        maincol <- 'blue'
        if(!flag) lines(time, cl, col=maincol, lwd=1.5, lty=2)
        lines(time, F, col=maincol, lwd=1.5)
        if(!flag) lines(time, cu, col=maincol, lwd=1.5, lty=2)
        if(!flag) lines(timep, clp, col=maincol, lty=2)
        lines(timep, Fp, col=maincol, lty=3)
        if(!flag) lines(timep, cup, col=maincol, lty=2)
        if(!flag & !'yearsepgrowth' %in% names(inp)) lines(timef, clf, col=rgb(0, 0, 1, 0.2))
        if(!flag & !'yearsepgrowth' %in% names(inp)) lines(timef, cuf, col=rgb(0, 0, 1, 0.2))
        lines(inp$time, Fmsyvec$msy, col='black')
        box(lwd=1.5)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.ffmsy
#' @title Plot estimated relative fishing mortality.
#' @details Plots estimated fishing mortality with Fmsy and associated confidence interval.
#' @param rep A result report as generated by running fit.spict.
#' @param logax Take log of y-axis? default: FALSE
#' @param main Title of plot.
#' @param ylim Limits for y-axis.
#' @param plot.obs If TRUE observations are plotted.
#' @param qlegend If TRUE legend explaining colours of observation data is plotted.
#' @param lineat Draw horizontal line at this y-value.
#' @param xlab Label of x-axis.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.ffmsy(rep)
#' @export
plotspict.ffmsy <- function(rep, logax=FALSE, main='Relative fishing mortality', ylim=NULL,
                            plot.obs=TRUE, qlegend=TRUE, lineat=1, xlab='Time',
                            stamp=get.version()){
    if(!'sderr' %in% names(rep)){
        log <- ifelse(logax, 'y', '')
        inp <- rep$inp
        cicol <- 'lightgray'
        qf <- get.par('logqf', rep, exp=TRUE)
        Fest <- get.par('logFs', rep, exp=TRUE)
        FF <- get.par('logFFmsy', rep, exp=TRUE)
        logFF <- get.par('logFFmsy', rep)
        Fmsy <- get.par('logFmsy', rep, exp=TRUE)
        Fmsyvec <- get.msyvec(inp, Fmsy)
        if(min(inp$dtc) < 1){
            # Annual    
            al1 <- annual(inp$time, logFF[, 1])
            al2 <- annual(inp$time, logFF[, 2])
            al3 <- annual(inp$time, logFF[, 3])
            inds <- which(!is.na(al1$annvec) & al2$anntime<=tail(inp$time[inp$indest],1))
            cl <- exp(al1$annvec[inds])
            cu <- exp(al3$annvec[inds])
            inds <- which(!is.na(al2$annvec) & al2$anntime<=tail(inp$time[inp$indest],1))
            time <- al1$anntime[inds]
            F <- exp(al2$annvec[inds])
            inds <- which(!is.na(al1$annvec) & al2$anntime>=tail(inp$time[inp$indest],1))
            clp <- exp(al1$annvec[inds])
            cup <- exp(al3$annvec[inds])
            inds <- which(!is.na(al2$annvec) & al2$anntime>=tail(inp$time[inp$indest],1))
            timep <- al1$anntime[inds]
            Fp <- exp(al2$annvec[inds])
            al1f <- annual(inp$time, logFF[, 1])
            al2f <- annual(inp$time, logFF[, 2])
            al3f <- annual(inp$time, logFF[, 3])
            inds <- which(!is.na(al1f$annvec))
            clf <- exp(al1f$annvec[inds])
            cuf <- exp(al3f$annvec[inds])
            inds <- which(!is.na(al2f$annvec))
            timef <- al2f$anntime[inds]
            Ff <- exp(al2f$annvec[inds])
        } else {
            time <- inp$time[inp$indest]
            cl <- FF[inp$indest, 1]
            F <- FF[inp$indest, 2]
            cu <- FF[inp$indest, 3]
            timep <- inp$time[inp$indpred]
            clp <- FF[inp$indpred, 1]
            Fp <- FF[inp$indpred, 2]
            cup <- FF[inp$indpred, 3]
            timef <- inp$time
            clf <- FF[, 1]
            Ff <- FF[, 2]
            cuf <- FF[, 3]
        }
        flag <- length(cu)==0 | all(!is.finite(cu))
        if(flag){
            fininds <- which(is.finite(Ff))
            if(length(ylim)!=2) ylim <- range(c(lineat, Ff[fininds]), na.rm=TRUE)
        } else {
            fininds <- which(apply(cbind(clf, cuf), 1, function(x) all(is.finite(x))))
            if(length(ylim)!=2) ylim <- range(c(lineat, cl[fininds], cu[fininds]), na.rm=TRUE)
        }
        ylim[2] <- min(c(ylim[2], 3*max(Ff[fininds]))) # Limit upper limit
        ylim <- c(min(ylim[1], lineat), max(ylim[2], lineat)) # Ensure that lineat is included in ylim
        #if(main==-1) main <- 'Relative fishing mortality'
        plot(timef, Ff, typ='n', main=main, ylim=ylim, col='blue', ylab=expression(F[t]/F[MSY]), xlab=xlab, xlim=range(c(inp$time, tail(inp$time, 1) + 0.5)), log=log)
        cicol2 <- rgb(0, 0, 1, 0.1)
        if(!flag) polygon(c(timef[fininds], rev(timef[fininds])), c(clf[fininds], rev(cuf[fininds])), col=cicol2, border=cicol2)
        if(min(inp$dtc) < 1){ # Plot estimated sub annual F 
            lines(inp$time, FF[, 2], col=rgb(0, 0, 1, 0.4))
        }
        abline(v=inp$time[inp$indlastobs], col='gray')
        if (plot.obs) plot.col(inp$timeE, inp$obsE/inp$dte*qf[2]/Fmsy[2], cex=0.7, do.line=FALSE, add=TRUE, add.legend=qlegend)
        if('true' %in% names(inp)){
            lines(inp$true$time, inp$true$Fs/inp$true$Fmsy, col=true.col()) # Plot true
        }
        maincol <- 'blue'
        lines(time, F, col=maincol, lwd=1.5)
        lines(timep, Fp, col=maincol, lty=3)
        if(!flag) lines(timef, clf, col=rgb(0, 0, 1, 0.2))
        if(!flag) lines(timef, cuf, col=rgb(0, 0, 1, 0.2))
        abline(h=lineat, col='black')
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        box(lwd=1.5)
        txt.stamp(stamp)
    }
}


#' @name plotspict.fb
#' @title Plot fishing mortality versus biomass.
#' @details Plots estimated fishing mortality as a function of biomass together with reference points and the prediction for next year given a constant F. The equilibrium biomass for F fixed to the current value is also plotted.
#' @param rep A result report as generated by running fit.spict.
#' @param logax Take log of x and y-axes? default: FALSE
#' @param plot.legend Plot legend explaining triangle.
#' @param ext Add relative level axis to top and right side.
#' @param rel.axes Plot axes in relative levels instead of absolute.
#' @param xlim Limits of x-axis.
#' @param ylim Limits of y-axis.
#' @param labpos Positions of time stamps of start and end points as in pos in text().
#' @param xlabel Label of x-axis. If NULL not used.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.fb(rep)
#' @export
plotspict.fb <- function(rep, logax=FALSE, plot.legend=TRUE, ext=TRUE, rel.axes=FALSE,
                         xlim=NULL, ylim=NULL, labpos=c(1, 1), xlabel=NULL, stamp=get.version()){
    if(!'sderr' %in% names(rep)){
        #omar <- par()$mar
        mar <- c(5.1, 4.3, 4.1, 4.1)
        if(dev.cur()==1){ # If plot is not open
            opar <- par(mar=mar)
            on.exit(par(opar))
        }
        if(dev.cur()==2){ # If plot is open, check if it is a 1x1 plot
            if(all(par()$mfrow == c(1, 1))){
                opar <- par(mar=mar)
                on.exit(par(opar))
            }
        }
        log <- ifelse(logax, 'xy', '')
        inp <- rep$inp
        Bmsyall <- get.par('logBmsy', rep, exp=TRUE)
        Fmsyall <- get.par('logFmsy', rep, exp=TRUE)
        Bmsy <- tail(Bmsyall, 1)
        Fmsy <- tail(Fmsyall, 1)
        if(rel.axes){
            ext <- FALSE
            bscal <- Bmsy[2]
            fscal <- Fmsy[2]
            xlab <- expression(B[t]/B[MSY])
            ylab <- expression(F[t]/F[MSY])
        } else {
            bscal <- 1
            fscal <- 1
            xlab <- expression(B[t])
            xlab <- add.catchunit(xlab, inp$catchunit)
            ylab <- expression(F[t])
        }
        Bp <- get.par('logBp', rep, exp=TRUE)
        Best <- get.par('logB', rep, exp=TRUE)
        logBest <- get.par('logB', rep)
        Fest <- get.par('logFs', rep, exp=TRUE)
        logFest <- get.par('logFs', rep)
        ns <- dim(Best)[1]
        Fp <- Fest[ns,]
        inds <- c(max(which(names(rep$value)=='logBmsy')), max(which(names(rep$value)=='logFmsy')))
        if (rep$opt$convergence == 0){
            cl <- try(make.ellipse(inds, rep))
        } else {
            cl <- numeric()
            class(cl) <- 'try-error'
        }
        if(class(cl) == 'try-error'){
            cl <- matrix(c(log(Bmsy[2]), log(Fmsy[2])), 1, 2)
        } 
        if(min(inp$dtc) < 1){ # Quarterly
            alb <- annual(inp$time, logBest[, 2])
            alf <- annual(inp$time, logFest[, 2])
            aind <- which(inp$time[inp$dtprediind] == alb$anntime)
            bbb <- exp(alb$annvec)/bscal
            fff <- exp(alf$annvec)/fscal
            fbtime <- alb$anntime
        } else {
            bbb <- Best[inp$indest,2]/bscal
            fff <- Fest[inp$indest,2]/fscal
            pind <- rep$inp$dtprediind
            fbtime <- inp$time[inp$indest]
        }
        Fl <- tail(unname(fff), 1)
        Bl <- tail(unname(bbb), 1)
        EBinf <- get.EBinf(rep)/bscal
        # Limits
        if(is.null(xlim)){
            xlim <- range(c(exp(cl[,1]), Best[,2], EBinf)/bscal, na.rm=TRUE)
            if(min(inp$dtc) < 1){
                # New annual limits
                xlim <- range(c(exp(alb$annvec), exp(cl[, 1]), EBinf)/bscal, na.rm=TRUE)
            }
            xlim[2] <- min(c(xlim[2], 8*Bmsy[2]/bscal), 2.2*max(bbb), na.rm=TRUE)
            xlim[2] <- max(c(xlim[2], Bmsy[2]/bscal), na.rm=TRUE)
        }
        if(is.null(ylim)){
            ylim <- range(c(exp(cl[,2]), Fest[,2])/fscal, na.rm=TRUE)
            if(min(inp$dtc) < 1){
                # New annual limits
                ylim <- range(c(exp(alf$annvec)/fscal, exp(cl[, 2])/fscal), na.rm=TRUE)
            }
            ylim[2] <- min(c(ylim[2], 8*Fmsy[2]/fscal), 2.2*max(fff), na.rm=TRUE)
            ylim[2] <- max(c(ylim[2], Fmsy[2]/fscal), na.rm=TRUE)
        }
        logminval <- 1e-4
        if(logax){ # Ensure that zero is not in the limit if taking log
            xlim[1] <- max(xlim[1], logminval)
            ylim[1] <- max(ylim[1], logminval)
        }
        # Plotting
        if(!is.null(xlabel)) xlab <- xlabel
        plot(Bmsy[2]/bscal, Fmsy[2]/fscal, typ='n', xlim=xlim, xlab=xlab, ylab=ylab, ylim=ylim, log=log)
        if(ext){
            if(logax){
                expx <- pretty(log10(xlim/Bmsy[2]))
                expy <- pretty(log10(ylim/Fmsy[2]))
                labx <- 10^expx
                laby <- 10^expy
            } else {
                labx <- pretty(xlim/Bmsy[2])
                laby <- pretty(ylim/Fmsy[2])
            }
            atx <- labx*Bmsy[2] #expression(10^expx)
            aty <- laby*Fmsy[2] #expression(10^expy)
            axis(3, labels=labx, at=atx)
            mtext(expression(B[t]/B[MSY]), side=3, las=0, line=2, cex=par('cex'))
            axis(4, labels=laby, at=aty)
            mtext(expression(F[t]/F[MSY]), side=4, las=0, line=2.5, cex=par('cex'))
        }
        alpha <- 0.15
        ymin <- ifelse(logax, logminval*1e-2, -10)
        xmin <- ifelse(logax, logminval*1e-2, xlim[1]-xlim[2])
        polygon(c(Bmsy[2]/bscal, Bmsy[2]/bscal, xlim[2]*2, xlim[2]*2), c(Fmsy[2], ymin, ymin, Fmsy[2])/fscal, col=rgb(0.5,0.8,0.4,1), border=NA) # Green
        yel <- rgb(1,0.925,0.55,1) # Yellow
        polygon(c(Bmsy[2]/bscal, Bmsy[2]/bscal, xmin, xmin), c(Fmsy[2], ymin, ymin, Fmsy[2])/fscal, col=yel, border=NA) # Yellow
        polygon(c(Bmsy[2]/bscal, Bmsy[2]/bscal, xlim[2]*2, xlim[2]*2), c(Fmsy[2], (ylim[2]+1)*2, (ylim[2]+1)*2, Fmsy[2])/fscal, col=yel, border=NA) # Yellow
        polygon(c(Bmsy[2]/bscal, Bmsy[2]/bscal, xmin, xmin), c(Fmsy[2], (ylim[2]+1)*2, (ylim[2]+1)*2, Fmsy[2])/fscal, col=rgb(1,0.188,0.188,1), border=NA) # Red
        abline(v=0, col='darkred', lty=2)
        cicol <- 'lightgray'
        cicolrgb <- col2rgb(cicol)/255
        cicoluse <- rgb(cicolrgb[1], cicolrgb[2], cicolrgb[3], 0.7)
        cicol2 <- 'gray'
        cicol2rgb <- col2rgb(cicol2)/255
        cicol2use <- rgb(cicol2rgb[1], cicol2rgb[2], cicol2rgb[3], 0.7)
        polygon(exp(cl[,1])/bscal, exp(cl[,2])/fscal, col=cicoluse, border=cicol2use)
        if('true' %in% names(inp)) points(inp$true$Bmsy/bscal, inp$true$Fmsy/fscal, pch=25, bg=true.col())
        maincol <- rgb(0,0,1,0.8)
        if(min(inp$dtc) < 1){
            lines(bbb, fff, col=maincol, lwd=1.5)
        } else {
            lines(bbb, fff, col=maincol, lwd=1.5)
            lines(Best[inp$indpred,2]/bscal, Fest[inp$indpred,2]/fscal, col=maincol, lty=3)
            Bll <- tail(Best[inp$indpred,2]/bscal, 1)
            Fll <- tail(Fest[inp$indpred,2]/fscal, 1)
            lines(c(Bll, EBinf), rep(Fll, 2), lwd=1.5, lty=3, col='blue')
            points(EBinf, Fll, pch=24, bg='gold')
        }
        nr <- length(inp$ini$logr)
        if(nr > 1) points(Bmsyall[1:(nr-1), 2]/bscal, Fmsyall[1:(nr-1), 2]/fscal, pch=24, bg='magenta')
        if(plot.legend){
            if(nr > 1){
                legend('topright', c('Current MSY', 'Previous MSY'), pch=3, col=c('black', 'magenta'), bg='white')
            } else {
                if('true' %in% names(inp)){
                    if(min(inp$dtc) < 1){
                        legend('topright', 'True', pch=25, pt.bg=true.col(), bg='white')
                    } else {
                        legend('topright', c(expression('E(B'[infinity]*')'), 'True'), pch=c(24, 25), pt.bg=c('gold', true.col()), bg='white')
                    }
                } else {
                    if(!min(inp$dtc) < 1){ legend('topright', expression('E(B'[infinity]*')'), pch=24, pt.bg='gold', bg='white')
                                       }
                }
            }
        }
        points(bbb[1], fff[1], pch=21, bg='white')
        text(bbb[1], fff[1], round(fbtime[1], 2), pos=labpos[1], cex=0.75, offset=0.5, xpd=TRUE)
        points(Bl, Fl, pch=22, bg='white')
        text(Bl, Fl, round(tail(fbtime, 1), 2), pos=labpos[2], cex=0.75, offset=0.5, xpd=TRUE)
        box(lwd=1.5)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.catch
#' @title Plot observed catch and predictions.
#' @details Plots observed catch and predictions using the current F and Fmsy. The plot also contains the equilibrium catch if the current F is maintained.
#' @param rep A result report as generated by running fit.spict.
#' @param main Title of plot.
#' @param ylim Limits for y-axis.
#' @param qlegend If TRUE legend explaining colours of observation data is plotted.
#' @param lcol Colour of prediction lines.
#' @param xlab Label of x-axis.
#' @param ylab Label of y-axis.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.catch(rep)
#' @export
plotspict.catch <- function(rep, main='Catch', ylim=NULL, qlegend=TRUE, lcol='blue',
                            xlab='Time', ylab=NULL, stamp=get.version()){
    if(!'sderr' %in% names(rep)){
        ylabflag <- is.null(ylab) # If null then not manually specified
        inp <- rep$inp
        ylimflag <- !is.null(ylim)
        Cscal <- 1
        cicol <- 'lightgray'
        MSY <- get.par('logMSY', rep, exp=TRUE)
        MSYvec <- get.msyvec(inp, MSY)
        Cpredest <- get.par('logCpred', rep, exp=TRUE)
        Cpredest[Cpredest<0] <- 0
        rep$Cp[rep$Cp<0] <- 0
        indest <- which(inp$timeCpred <= tail(inp$timeC,1))
        indpred <- which(inp$timeCpred >= tail(inp$timeC,1))
        dtc <- inp$dtcp
        if(min(inp$dtc) < 1){
            alo <- annual(inp$timeC, inp$obsC/inp$dtc)
            timeo <- alo$anntime
            obs <- alo$annvec
            al1 <- annual(inp$timeCpred[indest], Cpredest[indest, 1]/dtc[indest])
            al2 <- annual(inp$timeCpred[indest], Cpredest[indest, 2]/dtc[indest])
            al3 <- annual(inp$timeCpred[indest], Cpredest[indest, 3]/dtc[indest])
            inds <- which(!is.na(al2$annvec))
            time <- al2$anntime[inds]
            c <- al2$annvec[inds]
            cl <- al1$annvec[inds]
            cu <- al3$annvec[inds]
            al1p <- annual(inp$timeCpred[indpred], Cpredest[indpred, 1]/dtc[indpred])
            al2p <- annual(inp$timeCpred[indpred], Cpredest[indpred, 2]/dtc[indpred])
            al3p <- annual(inp$timeCpred[indpred], Cpredest[indpred, 3]/dtc[indpred])
            inds <- which(!is.na(al2p$annvec))
            timep <- al2p$anntime[inds]
            cp <- al2p$annvec[inds]
            clp <- al1p$annvec[inds]
            cup <- al3p$annvec[inds]
            al1f <- annual(inp$timeCpred, Cpredest[, 1]/dtc)
            al2f <- annual(inp$timeCpred, Cpredest[, 2]/dtc)
            al3f <- annual(inp$timeCpred, Cpredest[, 3]/dtc)
            inds <- which(!is.na(al2f$annvec))
            timef <- al2f$anntime[inds]
            clf <- al1f$annvec[inds]
            cf <- al2f$annvec[inds]
            cuf <- al3f$annvec[inds]
            if(any(inp$dtc==1)){
                inds <- which(inp$dtc==1)
                timeo <- c(timeo, inp$timeC[inds])
                obs <- c(obs, inp$obsC[inds])
                timeunsort <- c(time, inp$timeCpred[inds])
                timesort <- sort(timeunsort, index=TRUE)
                time <- timesort$x
                c <- c(c, Cpredest[inds, 2])[timesort$ix]
                cl <- c(cl, Cpredest[inds, 1])[timesort$ix]
                cu <- c(cu, Cpredest[inds, 3])[timesort$ix]
            }
        } else {
            timeo <- inp$timeC
            obs <- inp$obsC/inp$dtc
            time <- inp$timeCpred[indest]
            c <- Cpredest[indest, 2]/dtc[indest]
            cl <- Cpredest[indest, 1]
            cu <- Cpredest[indest, 3]
            timep <- inp$timeCpred[indpred]
            cp <- Cpredest[indpred, 2]/dtc[indpred]
            clp <- Cpredest[indpred, 1]
            cup <- Cpredest[indpred, 3]
            timef <- inp$timeCpred
            clf <- Cpredest[, 1]
            cf <- Cpredest[, 2]/dtc
            cuf <- Cpredest[, 3]
        }
        fininds <- which(apply(cbind(clf, cuf), 1, function(x) all(is.finite(x))))
        if(!ylimflag){
            if(length(ylim)!=2){
                ylim <- range(c(cl, cu, 0.9*MSY[1], 1.07*MSY[3]), na.rm=TRUE)/Cscal
                if(inp$dtpredc > 0) ylim <- range(ylim, clf[fininds], cuf[fininds])
            }
            ylim[2] <- min(c(ylim[2], 3*max(cf[fininds]))) # Limit upper limit
        }
        #if(main==-1) main <- 'Catch'
        if(ylabflag){
            ylab <- 'Catch'
            ylab <- add.catchunit(ylab, inp$catchunit)
        }
        plot(time, c, typ='n', main=main, xlab=xlab, ylab=ylab, xlim=range(c(inp$time, tail(inp$time,1))), ylim=ylim)
        polygon(c(inp$time, rev(inp$time)), c(MSYvec$ll,rev(MSYvec$ul)), col=cicol, border=cicol)
        cicol2 <- rgb(0, 0, 1, 0.1)
        lines(time, cl, col=lcol, lwd=1.5, lty=2)
        lines(time, cu, col=lcol, lwd=1.5, lty=2)
        abline(v=tail(inp$timeC,1), col='gray')
        plot.col(timeo, obs/Cscal, cex=0.7, do.line=FALSE, add=TRUE, add.legend=qlegend)
        # Highlight influential index observations
        if('infl' %in% names(rep) & min(inp$dtc) == 1){
            infl <- rep$infl$infl[1:inp$nobsC, ]
            cols <- apply(!is.na(infl), 1, sum)
            ncols <- length(unique(cols))
            inds <- which(cols>0)
            points(inp$timeC[inds], inp$obsC[inds]/Cscal, pch=21, cex=0.9, bg=cols[inds])
        }
        if('true' %in% names(inp)){
            abline(h=inp$true$MSY, col=true.col(), lty=1)
            abline(h=inp$true$MSY, col='black', lty=3)
        }
        lines(inp$time, MSYvec$msy)
        lines(time, c, col=lcol, lwd=1.5)
        if(inp$dtpredc > 0){
            lines(timep, cp, col=lcol, lty=3)
            lines(timep, clp, col=lcol, lwd=1, lty=2)
            lines(timep, cup, col=lcol, lwd=1, lty=2)
        }
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        box(lwd=1.5)
        txt.stamp(stamp)
    }
}


#' @name plotspict.production
#' @title Plot theoretical production curve and estimates.
#' @details Plots the theoretical production curve (production as a function of biomass) as calculated from the estimated model parameters. Overlaid is the estimated production/biomass trajectory.
#' @param rep A result report as generated by running fit.spict.
#' @param n.plotyears Plot years next to points if number of points is below n.plotyears. Default: 40.
#' @param main Title of plot.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.production(rep)
#' @export
plotspict.production <- function(rep, n.plotyears=40, main='Production curve', stamp=get.version()){
    if (!'sderr' %in% names(rep)){
        inp <- rep$inp
        Kest <- get.par('logK', rep, exp=TRUE)
        mest <- get.par('logm', rep, exp=TRUE)
        nr <- dim(mest)[1]
        gamma <- get.par('gamma', rep)
        n <- get.par('logn', rep, exp=TRUE)
        Bmsy <- get.par('logBmsy', rep, exp=TRUE)
        Bmsy <- c(1,1)
        nBplot <- 200
        Bplot <- seq(0.5*1e-8, Kest[2], length=nBplot)
        # Calculate production curve (Pst)
        pfun <- function(gamma, m, K, n, B) gamma*m/K*B*(1 - (B/K)^(n-1))
        Pst <- list()
        for (i in 1:nr) Pst[[i]] <- pfun(gamma[2], mest[i,2], Kest[2], n[2], Bplot)
        ylim <- c(0, max(unlist(Pst)/Bmsy[2], na.rm=TRUE))
        if (inp$reportall){
            Best <- get.par('logB', rep, exp=TRUE)
            Pest <- get.par('P', rep)
            Bplot <- seq(0.5*min(c(1e-8, Best[, 2])), 1*max(c(Kest[2], Best[, 2])), length=nBplot)
            for(i in 1:nr) Pst[[i]] <- pfun(gamma[2], mest[i,2], Kest[2], n[2], Bplot)
            Bvec <- Best[inp$ic[1:dim(Pest)[1]], 2]
            xlim <- range(Bvec/Kest[2], 0, 1)
            ylim <- c(min(0, Pest[,2]/Bmsy[2]), max(Pest[,2]/Bmsy[2], unlist(Pst)/Bmsy[2], na.rm=TRUE))
        } else {
            xlim <- range(Bplot/Kest[2], na.rm=TRUE)
        }
        dt <- inp$dt[-1]
        inde <- inp$indest[-length(inp$indest)]
        indp <- inp$indpred[-1]-1
        ylab <- 'Production'
        ylab <- add.catchunit(ylab, inp$catchunit)
        #if(main==-1) main <- 'Production curve'
        plot(Bplot/Kest[2], Pst[[nr]]/Bmsy[2], typ='l', ylim=ylim, xlim=xlim, xlab='B/K', ylab=ylab, col=1, main=main)
        if (nr > 1) for(i in 1:(nr-1)) lines(Bplot/Kest[2], Pst[[i]]/Bmsy[2], col='gray')
        if (inp$reportall){
            lines(Bvec/Kest[2], Pest[, 2]/Bmsy[2], col=4, lwd=1.5)
            points(Bvec/Kest[2], Pest[, 2]/Bmsy[2], col=4, pch=20, cex=0.7)
            par(xpd=TRUE)
            if (length(inp$ic) < n.plotyears){
                inds <- c(1, length(Bvec), seq(1, length(Bvec), by=2))
                labs <- round(inp$time[inp$ic], 2)
                text(Bvec[inds]/Kest[2], Pest[inds, 2]/Bmsy[2], labels=labs[inds], cex=0.75, pos=4, offset=0.25)
            }
            par(xpd=FALSE)
        }
        mx <- (1/n[2])^(1/(n[2]-1))
        abline(v=mx, lty=3)
        abline(h=0, lty=3)
        box(lwd=1.5)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.tc
#' @title Plot time constant.
#' @details Plots the time required for the biomass to reach a certain proportion of Bmsy. The time required to reach 95\% of Bmsy is highlighted.
#' @param rep A result report as generated by running fit.spict.
#' @param main Title of plot.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plotspict.tc(rep)
#' @export
plotspict.tc <- function(rep, main='Time to Bmsy', stamp=get.version()){
    if(!'sderr' %in% names(rep) & rep$opt$convergence == 0){
        inp <- rep$inp
        B0cur <- get.par('logBl', rep, exp=TRUE)[2]
        Kest <- get.par('logK', rep, exp=TRUE)
        m <- get.par('logm', rep, exp=TRUE)
        mmean <- tail(m, 1)[2]
        n <- get.par('logn', rep, exp=TRUE)
        gamma <- calc.gamma(n[2])
        sdbest <- get.par('logsdb', rep, exp=TRUE)
        Fmsy <- tail(get.par('logFmsy', rep, exp=TRUE), 1)
        Bmsy <- tail(get.par('logBmsy', rep, exp=TRUE), 1)
        if(B0cur < Bmsy[2]){
            do.flag <- ifelse(B0cur/Bmsy[2]>0.95, FALSE, TRUE)
        }
        if(B0cur > Bmsy[2]){
            do.flag <- ifelse(Bmsy[2]/B0cur>0.95, FALSE, TRUE)
        }
        if(do.flag){
            if(B0cur < Bmsy[2]) facvec <- c(0, 0.75, 0.95, 1)
            if(B0cur > Bmsy[2]) facvec <- c(2, 1.25, 1.05, 1)
            cols <- c('green3', 'blue', 'red', 'orange', 5:8)
            Fvec <- round(facvec*Fmsy[2], digits=4)
            nFvec <- length(Fvec)
            g <- function(F, K, m, n, sdb, B0, dt){
                return(exp( log(B0) + (gamma*m/K - gamma*m/K*(B0/K)^(n-1) - F - 0.5*sdb^2)*dt ))
            }
            simdt <- 0.01
            nt <- 10000
            Bsim <- matrix(0, nFvec, nt)
            time <- matrix(0, nFvec, nt)
            for(i in 1:nFvec){
                time[i, ] <- seq(0, simdt*(nt-1), by=simdt)
                Bsim[i, ] <- rep(0, nt)
                Bsim[i, 1] <- B0cur
                for(j in 2:nt){
                    Bsim[i, j] <- g(Fvec[i], Kest[2], mmean, n[2], sdbest[2], Bsim[i, j-1], simdt)
                }
            }
            Bsim <- Bsim/Bmsy[2]
            frac <- 0.95
            if(B0cur < Bmsy[2]) inds <- which(Bsim[nFvec, ]<0.99)
            if(B0cur > Bmsy[2]) inds <- which(Bsim[nFvec, ]>(1/0.99))
            ylim <- range(Bsim[nFvec, ], na.rm=TRUE)
            xlim <- range(time[nFvec, inds])
            xlim[2] <- min(xlim[2], 15) # Max 15 years ahead
            plot(time[1, ], Bsim[1, ], typ='l', xlim=xlim, ylim=ylim, col=cols[1],
                 ylab='Proportion of Bmsy', xlab='Years to Bmsy', main=main, lwd=1.5)
            abline(h=c(frac, 1/frac), lty=1, col='lightgray')
            abline(h=1, lty=3)
            for(i in 2:nFvec) lines(time[i, ], Bsim[i, ], col=cols[i], lwd=1.5)
            vt <- rep(0, nFvec)
            if(B0cur < Bmsy[2]) for(i in 1:nFvec) vt[i] <- time[i, max(which(Bsim[i, ]<frac))]
            if(B0cur > Bmsy[2]) for(i in 1:nFvec) vt[i] <- time[i, max(which(1/Bsim[i, ]<frac))]
            for(i in 1:nFvec) abline(v=vt[i], col=cols[i], lty=2)
            lgnplace <- 'bottomright'
            if(B0cur > Bmsy[2]) lgnplace <- 'topright'
            legend(lgnplace, legend=paste('F =',facvec,'x Fmsy'), lty=1, col=cols[1:nFvec],
                   lwd=rep(1.5,nFvec), bg='white')
            points(vt, rep(par('usr')[3], nFvec), col=cols[1:nFvec], pch=4)
            box(lwd=1.5)
            if (rep$opt$convergence != 0){
                warning.stamp()
            }
            txt.stamp(stamp)
        }
    }
}


#' @name plotspict.season
#' @title Plot the mean F cycle
#' @details If seasonal data are available the seasonal cycle in the fishing mortality can be estimated. This function plots this mean F cycle.
#' @param rep A result report as generated by running fit.spict.
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @export
plotspict.season <- function(rep, stamp=get.version()){
    if(!'par.fixed' %in% names(rep)) stop('Input object was not a valid output from fit.spict()!')
    if(!'sderr' %in% names(rep) & 'logphi' %in% names(rep$par.fixed)){
        jan <- as.POSIXct("2015-01-01 00:00:01 UTC", tz='UTC')
        apr <- jan+(31+28+31)*24*60*60
        jul <- apr+(30+31+30)*24*60*60
        oct <- jul+(31+31+30)*24*60*60
        logF <- get.par('logF', rep)
        meanlogF <- mean(logF[, 2])
        meanlogF <- 0 # Don't add mean F
        logphi <- get.par('logphi', rep)
        naflag <- any(is.na(logphi[, 1]))
        inds <- which(colnames(rep$cov.fixed) == 'logphi')
        ssf <- get.par('seasonsplinefine', rep)
        covmat <- rep$cov.fixed[inds, inds]
        seasonsplineest <- get.spline(logphi[, 2], order=rep$inp$splineorder, dtfine=rep$inp$dteuler)
        test <- seq(0, 1, length=length(seasonsplineest))
        yest <- exp(meanlogF + seasonsplineest)
        seasonsplinesmoo <-  ssf[, 2]
        sssl <- exp(meanlogF + ssf[, 1])
        sssu <- exp(meanlogF + ssf[, 3])
        nsss <- length(seasonsplinesmoo)
        t <- seq(0, 1, length=nsss)
        y <- exp(meanlogF + seasonsplinesmoo)
        ylim <- range(yest, y)
        if(!naflag) ylim <- range(yest, y, sssl, sssu)
        if("true" %in% names(rep$inp)){
            if(rep$inp$true$seasontype==1){
                seasonsplinetrue <- get.spline(rep$inp$true$logphi, order=rep$inp$true$splineorder, dtfine=rep$inp$true$dteuler)
                ttrue <- seq(0, 1, length=length(seasonsplinetrue))
                ytrue <- exp(seasonsplinetrue) # Don't add mean F
                ylim <- range(c(yest, y, ytrue))
            }
        }
        plot(t, y, typ='n', xaxt='n', xlab='Time of year', ylab='Seasonal spline', main=paste('Spline order:',rep$inp$splineorder), ylim=ylim)
        cicol2 <- rgb(0, 0, 1, 0.1)
        cicol3 <- rgb(0, 0, 1, 0.2)
        polygon(c(t, rev(t)), c(sssl, rev(sssu)), col=cicol2, border=cicol2)
        if(!naflag){
            lines(t, sssl, col=cicol3)
            lines(t, sssu, col=cicol3)
        }
        lab <- strftime(c(jan, apr, jul, oct, jan), format='%b')
        ats <- c(0, 0.25, 0.5, 0.75, 1)
        abline(v=ats, lty=3, col='lightgray')
        ats2 <- pretty(c(sssl, sssu))
        if("true" %in% names(rep$inp)) ats2 <- pretty(c(sssl, sssu, ytrue))
        abline(h=ats2, lty=3, col='lightgray')
        lines(t, y, lwd=1, col='green')
        lines(test, yest, lwd=1.5, col=4, typ='s')
        if("true" %in% names(rep$inp)){
            if(rep$inp$true$seasontype==1){
                lines(ttrue, ytrue, lwd=1, col=true.col(), typ='s')
            }
        }
        axis(1, at=ats, labels=lab)
        box(lwd=1.5)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.btrend
#' @title Plot the expected biomass trend
#' @param rep A result report as generated by running fit.spict.
#' @return Nothing.
#' @export
plotspict.btrend <- function(rep){
    if(!'sderr' %in% names(rep)){
        Bind <- get.par('Bind', rep)
        B <- get.par('logB', rep, exp=TRUE)
        plot(rep$inp$time, Bind[, 2], typ='l', ylim=c(-2, 2), xlab='Time', ylab='B trend')
        lines(rep$inp$time, Bind[, 1], lty=2)
        lines(rep$inp$time, Bind[, 3], lty=2)
        lines(rep$inp$time[-1], diff(B[, 2]), col=4, lwd=1.5)
        abline(h=0, col='gray')
        legend('topleft', legend=c('Expected', 'Observed'), lty=1, lwd=c(1, 1.5), col=c(1, 4))
        box(lwd=1.5)
        if (rep$opt$convergence != 0){
            warning.stamp()
        }
    }
}


#' @name plot.spictcls
#' @title Plot summarising spict results.
#' @details Create a plot containing the following:
#' \itemize{
#'  \item{1. Estimated biomass using plotspict.biomass().}
#'  \item{2. Estimated fishing mortality using plotspict.f().}
#'  \item{3. Observed versus predicted catches using plotspict.catch().}
#'  \item{4. Estimated biomass relative to Bmsy using plotspict.bbmsy().}
#'  \item{5. Estimated fishing mortality relative to Fmsy using plotspict.ffmsy().}
#'  \item{6. Estimated F versus estimated B using plotspict.fb().}
#'  \item{7. Observed versus theoretical production using plotspict.production().}
#' }
#'
#' Optional plots included if relevant:
#' \itemize{
#'  \item{ Estimated seasonal spline using plotspict.season().}
#'  \item{ Calculated time-constant using plotspict.tc().}
#'  \item{ First prior and corresponding posterior distribution using plotspict.priors().}
#'  \item{ One-step-ahead residuals of catches using plotspict.osar().}
#'  \item{ One-step-ahead residuals of catches using plotspict.osar().}
#' }
#' 
#' @param x A result report as generated by running fit.spict.
#' @param ... additional arguments affecting the summary produced.
#' @return Nothing.
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' plot(rep)
#' @export
plot.spictcls <- function(x, ...){
    rep <- x
    logax <- FALSE # Take log of relevant axes? default: FALSE
    if (!exists('stamp')){
        stamp <- get.version()
    }
    if('par.fixed' %in% names(rep) & rep$inp$do.sd.report){
        inp <- rep$inp
        if(inp$reportall){
            #dev.new(width=10, height=10)
            if('osar' %in% names(rep)){
                opar <- par(mfrow=c(4, 3), oma=c(0.2, 0.2, 0, 0), mar=c(5,4,2.5,3.5))
            } else {
                opar <- par(mfrow=c(3, 3), oma=c(0.2, 0.2, 0, 0), mar=c(5,4,2.5,3.5))
            }
            # Biomass
            plotspict.biomass(rep, logax=logax, stamp='')
            # F
            plotspict.f(rep, logax=logax, qlegend=FALSE, stamp='')
            # Catch
            plotspict.catch(rep, qlegend=FALSE, stamp='')
            # B/Bmsy
            plotspict.bbmsy(rep, logax=logax, qlegend=FALSE, stamp='')
            # F/Fmsy
            plotspict.ffmsy(rep, logax=logax, qlegend=FALSE, stamp='')
            # F versus B
            plotspict.fb(rep, logax=logax, plot.legend=TRUE, stamp='')
        } else {
            cat('inp$reportall = FALSE so not much to plot.\n')
            if('osar' %in% names(rep)){
                opar <- par(mfrow=c(3, 2), oma=c(0.2, 0.2, 0, 0), mar=c(5,4,2,3.5))
            } else {
                opar <- par(mfrow=c(2, 1), oma=c(0.2, 0.2, 0, 0), mar=c(5,4,2,3.5))
            }
            on.exit(opar)
        }
        # Production curve
        plotspict.production(rep, stamp='')
        # Seasonal F
        if(inp$nseasons > 1 & inp$seasontype==1){
            plotspict.season(rep, stamp='')
        }
        # Time constant
        if(inp$nseasons == 1){
            plotspict.tc(rep, stamp='')
        }
        # Priors
        if('priors' %in% names(rep$inp)){
            plotspict.priors(rep, do.plot=1, stamp='')
        }
        if('osar' %in% names(rep)){
            plotspict.osar(rep, qlegend=FALSE)
        }
        if(inp$reportall){
            if('infl' %in% names(rep)){
                # Plot influence summary
                plotspict.inflsum(rep, stamp='')
            }
        }
    } else {
        if('inp' %in% names(rep)){
            plotspict.ci(rep$inp, stamp='')
        } else {
            stop('Nothing to plot!')
        }
    }
    txt.stamp(stamp)
}


#' @name put.ax
#' @title Adds the x-axis to influence plots
#' @details TBA
#' @param rep A valid result from calc.influence().
#' @return Nothing.
put.xax <- function(rep){
    inp <- rep$inp
    sernames <- colnames(rep$infl$dosarpvals)
    xs <- c(inp$nobsC, inp$nobsI)
    xat <- cumsum(xs[-length(xs)]+0.5)
    xmid <- c(0, xat) + xs/2
    axis(1, at=xat, labels='')
    axis(1, at=xmid, labels=sernames, tick=FALSE)
    abline(v=xat, lty=2, col='gray')
}


#' @name plotspict.infl
#' @title Plots influence statistics of observations.
#' @details TBA
#' @param rep A valid result from calc.influence().
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @export
plotspict.infl <- function(rep, stamp=get.version()){
    inp <- rep$inp
    #sernames <- c('C', paste0('I', 1:inp$nindex))
    dfbeta <- rep$infl$dfbeta
    dpar <- rep$infl$dpar
    ddetcov <- rep$infl$ddetcov
    dosarpvals <- rep$infl$dosarpvals
    sernames <- colnames(dosarpvals)
    nser <- inp$nindex+1
    nobs <- inp$nobsC + sum(inp$nobsI)
    parnams <- colnames(dfbeta)
    np <- length(parnams)
    rwnms <- rownames(dfbeta)
    infl <- rep$infl$infl
    opar <- par(mfrow=c(2,2))
    on.exit(par(opar))
    # Plot covratio
    dl <- 3*length(rep$par.fixed)/nobs
    covratio <- exp(ddetcov)
    ac <- abs(covratio-1)
    inds <- which(ac > dl)
    plot(ac, ylab='abs(covratio-1)', ylim=c(0, 1.05*max(ac)), xlab='', xaxt='n', main='COVRATIO')
    abline(h=dl)
    text(inds, ac[inds], rwnms[inds], pos=3, cex=0.7)
    put.xax(rep)
    box(lwd=1.5)
    # Plot OSAR p-values
    osarpvals <- get.osar.pvals(rep)
    alpha <- 0.05
    orgres <- osarpvals < alpha
    newres <- dosarpvals < alpha
    chgmat <- matrix(0, nobs, nser)
    for(i in 1:nser) chgmat[, i] <- newres[, i]-orgres[i]
    inds <- which(chgmat!=0)
    rc <- arrayInd(inds, dim(dosarpvals))
    nms <- rwnms[rc[, 1]]
    plot(dosarpvals[, 1], ylab='p-value', ylim=0:1, xlab='', xaxt='n', main='OSAR p-values')
    for(i in 2:nser) points(dosarpvals[, i], col=i)
    abline(h=osarpvals, col=1:nser)
    for(i in 1:length(inds)){
        text(rc[i, 1], dosarpvals[rc[i, 1], rc[i, 2]], nms[i], pos=3, cex=0.7, col=rc[i, 2])
    }
    abline(h=0.05, lwd=2, lty=3)
    legend('topleft', legend=sernames, pch=1, col=1:nser)
    put.xax(rep)
    box(lwd=1.5)
    # Plot dfbeta
    adfbeta <- abs(dfbeta)
    al <- 2/sqrt(nobs)
    inds <- which(adfbeta>al)
    rc <- arrayInd(inds, dim(adfbeta))
    nms <- rwnms[rc[, 1]]
    plot(adfbeta[, 1], ylim=c(0, 1.05*max(adfbeta)), ylab='abs dfbeta', xlab='', xaxt='n', main='DFBETA')
    for(i in 2:np) points(adfbeta[, i], col=i)
    abline(h=al, lwd=2, lty=3)
    for(i in 1:length(inds)){
        text(rc[i, 1], adfbeta[rc[i, 1], rc[i, 2]], nms[i], pos=3, cex=0.7, col=rc[i, 2])
        text(rc[i, 1], adfbeta[rc[i, 1], rc[i, 2]], round(dpar[rc[i, 1], rc[i, 2]], 3), pos=1, cex=0.7, col=rc[i, 2])
    }
    legend('topleft', legend=parnams, pch=1, col=1:np)
    put.xax(rep)
    box(lwd=1.5)
    # Plot of influence
    plotspict.inflsum(rep)
    box(lwd=1.5)
    txt.stamp(stamp)
}


#' @name plotspict.inflsum
#' @title Plots summary of influence statistics of observations.
#' @details TBA
#' @param rep A valid result from calc.influence().
#' @param stamp Stamp plot with this character string.
#' @return Nothing.
#' @export
plotspict.inflsum <- function(rep, stamp=get.version()){
    infl <- rep$infl$infl
    nobs <- dim(infl)[1]
    ninfl <- dim(infl)[2]
    matplot(infl, pch=1, col=1, yaxt='n', ylab='', xlab='', xaxt='n', main='Overall influence', type='n')
    axis(2, at=1:ninfl, labels=colnames(infl))
    cols <- apply(!is.na(infl), 1, sum)
    for(i in 1:nobs) abline(v=i, col=cols[i], lwd=(1+cols[i]/5))
    matplot(infl, pch=1, col=1, add=TRUE)
    put.xax(rep)
    txt.stamp(stamp)
}


#' @name plotspict.likprof
#' @title Plots result of likelihood profiling.
#' @details TBA
#' @param input Result of running likprof.spict().
#' @param logpar If TRUE log of parameters are shown.
#' @param stamp Stamp plot with this character string.
#' @return Nothing but shows a plot.
#' @export
plotspict.likprof <- function(input, logpar=FALSE, stamp=get.version()){
    repflag <- 'par.fixed' %in% names(input)
    if(repflag){ # This is a result of fit.spict
        rep <- input
        inp <- rep$inp
        nll <- rep$opt$objective
    } else { # This is an input list to fit.spict without estimates
        inp <- input
        nll <- min(inp$likprof$likvals)
    }
    likprof <- inp$likprof
    np <- length(likprof$pars)
    pv <- likprof$parvals
    pars <- likprof$pars
    loginds <- grep('log', pars)
    expinds <- setdiff(1:np, loginds)
    if(!logpar){
        pv[, loginds] <- exp(pv[, loginds])
        pars[loginds] <- gsub('log', '', pars[loginds])
    } else {
        pv[, expinds] <- log(pv[, expinds])
        pars[expinds] <- paste0('log', pars[expinds])
    }
    if(np==1){
        plot(pv[, 1], likprof$likvals, typ='l', xlab=pars[1], ylab='Objective function', lwd=1.5)
        lrlim <- 0.5*qchisq(0.95, 1) + nll
        abline(h=lrlim, lty=2) # Significance level
        pn <- likprof$pars
        par <- get.par(pn, rep) # in log
        pvq <- seq(min(likprof$parvals[, 1]), max(likprof$parvals[, 1]), length=100)
        quadapprox <- -dnorm(pvq, par[2], par[4], log=TRUE)
        quadapprox <- quadapprox + min(likprof$likvals) - min(quadapprox)
        lines(pvq, quadapprox, lty=3)
        legend('topleft', c('95% signf. lvl.', 'Quadratic approx.'), lty=2:3, bg='white')
    } else {
        pvals <- pchisq(2*(likprof$likvals - nll), np)
        contour(pv[, 1], pv[, 2], pvals, xlab=pars[1], ylab=pars[2], levels=c(0.5, 0.8, 0.95))
    }
    box(lwd=1.5)
    txt.stamp(stamp)
}


#' @name plotspict.retro
#' @title Plot results of retrospective analysis 
#' @param rep A valid result from fit.spict.
#' @param stamp Stamp plot with this character string.
#' @return Nothing
#' @export
plotspict.retro <- function(rep, stamp=get.version()){
    opar <- par(mfrow=c(2, 2), oma=c(1, 0.2, 0, 2), mar=c(4,4,2,1))
    on.exit(par(opar))
    nretroyear <- length(rep$retro)
    bs <- list()
    for(i in 1:nretroyear) bs[[i]] <- get.par('logB', rep$retro[[i]], exp=TRUE)[rep$retro[[i]]$inp$indest, 2]
    bbs <- list()
    for(i in 1:nretroyear) bbs[[i]] <- get.par('logBBmsy', rep$retro[[i]], exp=TRUE)[rep$retro[[i]]$inp$indest, 2]
    fs <- list()
    for(i in 1:nretroyear) fs[[i]] <- get.par('logF', rep$retro[[i]], exp=TRUE)[rep$retro[[i]]$inp$indest, 2]
    ffs <- list()
    for(i in 1:nretroyear) ffs[[i]] <- get.par('logFFmsy', rep$retro[[i]], exp=TRUE)[rep$retro[[i]]$inp$indest, 2]
    time <- list()
    for(i in 1:nretroyear) time[[i]] <- rep$retro[[i]]$inp$time[rep$retro[[i]]$inp$indest]

    # Do plots
    par(mfrow=c(2, 2))
    plot(time[[1]], bs[[1]], typ='l', ylim=range(unlist(bs)), xlab='Time', ylab = expression(B[t]), lwd=1.5)
    for(i in 2:nretroyear) lines(time[[i]], bs[[i]], col=i, lwd=1.5)
    box(lwd=1.5)
    plot(time[[1]], fs[[1]], typ='l', ylim=range(unlist(fs)), xlab='Time', ylab = expression(F[t]), lwd=1.5)
    for(i in 2:nretroyear) lines(time[[i]], fs[[i]], col=i, lwd=1.5)
    box(lwd=1.5)
    plot(time[[1]], bbs[[1]], typ='l', ylim=range(unlist(bbs)), xlab='Time', ylab = expression(B[t]/B[MSY]), lwd=1.5)
    for(i in 2:nretroyear) lines(time[[i]], bbs[[i]], col=i, lwd=1.5)
    box(lwd=1.5)
    plot(time[[1]], ffs[[1]], typ='l', ylim=range(unlist(ffs)), xlab='Time', ylab = expression(F[t]/F[MSY]), lwd=1.5)
    for(i in 2:nretroyear) lines(time[[i]], ffs[[i]], col=i, lwd=1.5)
    box(lwd=1.5)
    txt.stamp(stamp)
}


#' @name plotspict.ci
#' @title Plot catch and index data.
#' @param inp An input list containing data.
#' @param stamp Stamp plot with this character string.
#' @return Nothing
#' @export
plotspict.ci <- function(inp, stamp=get.version()){
    #op <- par()
    inp <- check.inp(inp)
    if (sum(inp$nobsI) > 0){
        # Remove effort observations as they are not used in this plot
        inp$obsE <- NULL
        inp$timeE <- NULL
        c <- guess.m(inp, all.return=TRUE)
        time <- inp$timeC
        y <- inp$obsC
        z <- inp$obsI[[1]]
        mfrow <- c(3, 2)
        if(class(c) == 'list'){ # A regression line could be fitted
            MSY <- c$MSY
            Emsy <- c$Emsy
            a <- c$a
            b <- c$b
            x <- c$x
            mod0 <- c$mod0
            xlim <- Re(polyroot(c(0, a, b)))
            xp <- data.frame('x'=seq(xlim[1], xlim[2], length=100))
            yp <- a*xp$x + b*xp$x^2 # Dome
            yp0 <- predict(mod0, xp)
        } else {
            MSY <- c
            mfrow <- c(1, 2)
        }
        if(inp$nseasons > 1){
            if(inp$nindex %in% 1) mfrow <- c(2, 2)
            if(inp$nindex %in% 2:3) mfrow <- c(2, 3)
            if(inp$nindex %in% 4:5) mfrow <- c(2, 4)
        }
        opar <- par(mfrow=mfrow)
        on.exit(par(opar))
        plot.seasondiff <- function(time, obs, ylab='Obs'){
            dt <- time[-length(time)]
            dy <- diff(log(obs))
            plot(dt, dy, typ='n', col='lightgray', ylab=ylab, xlab='Time')
            plot.col(dt, dy, do.line=FALSE, add=TRUE, typ='l')
            abline(h=0, lty=2, col='gray')
            box(lwd=1.5)
        }
        # Plot data
        plotspict.data(inp, MSY=MSY, one.index=1, stamp='')
        # Plot seasonal patterns
        if(inp$nseasons > 1){
            plot.seasondiff(inp$timeC, y, ylab='diff log catch')
            for(i in 1:inp$nindex) plot.seasondiff(inp$timeI[[i]], inp$obsI[[i]], ylab='diff log index 1')
        }
        if(class(c) == 'list' & inp$nseasons == 1){
            #
            plot(x, z, typ='l', xlim=xlim, ylab='Index', xlab='Catch/Index (E, effort proxy)', main=paste('R-squared:', round(summary(mod0)$r.squared, 3)), ylim=range(0, a, z))
            plot.col(time, z, obsx=x, do.line=FALSE, add=TRUE)
            lines(xp$x, yp0, col=4)
            box(lwd=1.5)
            #
            plot(x, y, typ='l', xlim=xlim, ylim=range(0, y, yp), ylab='Catch', xlab='Catch/Index (E, effort proxy)', main=paste('Emsy guess:', round(Emsy, 3)))
            plot.col(time, y, obsx=x, do.line=FALSE, add=TRUE)
            abline(h=MSY, lty=2)
            lines(xp$x, yp, col=4)
            box(lwd=1.5)
            #
            plot(z, y, typ='b', ylab='Catch', xlab='Index')
            plot.col(time, y, obsx=z, do.line=FALSE, add=TRUE)
            abline(h=MSY, lty=2)
            box(lwd=1.5)
            #
            plot(y[-length(y)], diff(z)/z[-length(z)], typ='b', xlab='Catch', ylab='Prop. incr. in index')
            plot.col(time, diff(z)/z[-length(z)], obsx=y[-length(y)], do.line=FALSE, add=TRUE)
            abline(h=0, lty=2)
            abline(v=MSY, lty=2)
            box(lwd=1.5)
        }
    } else {
        plotspict.data(inp)
    }
    txt.stamp(stamp)
    #par(op)
}


#' @name plotspict.priors
#' @title Plot priors and posterior distribution.
#' @param rep A result from fit.spict.
#' @param do.plot Integer defining maximum number of priors to plot.
#' @param stamp Stamp plot with this character string.
#' @return Nothing
#' @export
plotspict.priors <- function(rep, do.plot=4, stamp=get.version()){
    inp <- rep$inp
    npriors <- length(inp$priors)
    useflags <- numeric(npriors)
    for(i in 1:npriors){
        useflag <- inp$priors[[i]][3]
        nm <- names(inp$priors)[i]
        phase <- 1
        if(nm %in% names(inp$phases)) phase <- inp$phases[[nm]]
        if(phase > 0){ # Avoid plotting priors of parameters that are fixed
            useflags[i] <- inp$priors[[i]][3]
        } else {
            useflags[i] <- 0
        }
    }
    inds <- which(useflags==1)
    ninds <- length(inds)
    ninds <- min(ninds, do.plot)
    nused <- sum(useflags)
    if(ninds > 0){
        for(i in 1:ninds){
            j <- inds[i]
            priorvec <- inp$priors[[j]]
            nm <- names(inp$priors)[j]
            nmpl <- sub('log', '', nm)
            par <- get.par(nm, rep, exp=FALSE)
            repriors <- c('logB', 'logF', 'logBBmsy', 'logFFmsy')
            if(nm %in% repriors){
                par <- par[priorvec[5], ]
                nmpl <- paste0(nmpl, fd(priorvec[4]))
                if(nm == 'logB') nmpl <- add.catchunit(nmpl, inp$catchunit)
            }
            for (rr in 1:nrow(par)){
                nmpl <- sub('log', '', nm)
                if (nrow(par) > 1) nmpl <- paste0(nmpl, rr)
                mu <- ifelse(is.na(par[rr, 4]), priorvec[1], par[rr, 2])
                sd <- ifelse(is.na(par[rr, 4]), priorvec[2], par[rr, 4])
                xmin <- mu - 3*sd
                xmax <- mu + 3*sd
                x <- seq(xmin, xmax, length=200)
                priorvals <- dnorm(x, priorvec[1], priorvec[2])
                if (is.na(par[rr, 4])){
                    posteriorvals <- NULL
                } else {
                    posteriorvals <- dnorm(x, par[rr, 2], par[rr, 4])
                }
                plot(exp(x), priorvals, typ='l', xlab=nmpl, ylab='Density', log='x',
                     lwd=1.5, ylim=c(0, max(priorvals, posteriorvals)*1.3))
                if (is.na(par[rr, 4])){
                    if (!is.na(par[rr, 2])) abline(v=exp(par[rr, 2]), lty=2, col=3, lwd=1.5)
                    legend('topright', legend=c('Prior', 'Post. Mean'), lty=1:2,
                           col=c(1, 3), lwd=1.5)
                } else {
                    lines(exp(x), posteriorvals, col=3, lwd=1.5)
                    legend('topright', legend=c('Prior', 'Post.'), lty=1,
                           col=c(1, 3), lwd=1.5)
                }
                box(lwd=1.5)
                if (rep$opt$convergence != 0){
                    warning.stamp()
                }
            }
        }
        txt.stamp(stamp)
    }
}


#' @name plotspict.data
#' @title Plot input data
#' @param inpin An input list containing data.
#' @param MSY Value of MSY.
#' @param one.index Integer indicating the number of the index to plot.
#' @param qlegend
#' @param stamp Stamp plot with this character string.
#' @return Nothing
#' @export
plotspict.data <- function(inpin, MSY=NULL, one.index=NULL, qlegend=FALSE, stamp=get.version()){
    inp <- check.inp(inpin)
    #nseries <- inp$nindex + 1 + as.numeric(inp$nobsE > 0)
    nseries <- inp$nseries
    if(nseries %in% 1:2) mfrow <- c(2, 1)
    if(nseries %in% 3:4) mfrow <- c(2, 2)
    if(nseries %in% 5:6) mfrow <- c(2, 3)
    if(nseries %in% 7:9) mfrow <- c(3, 3)
    if(dev.cur()==1){
        opar <- par(mfrow=mfrow)
        on.exit(par(opar))
    } else {
        if(sum(par()$mfrow)==2){
            opar <- par(mfrow=mfrow)
            on.exit(par(opar))
        }
    }
    xlim <- range(inp$timeC, unlist(inp$timeI), inp$timeE)
    # Plot catch
    main <- paste0('Nobs C: ', inp$nobsC)
    ylab <- 'Catch'
    ylab <- add.catchunit(ylab, inp$catchunit)
    plot(inp$timeC, inp$obsC, typ='l', ylab=ylab, xlab='Time', main=main, xlim=xlim)
    grid()
    plot.col(inp$timeC, inp$obsC, do.line=FALSE, cex=0.6, add=TRUE, add.legend=qlegend)
    if(!is.null(MSY)) abline(h=MSY, lty=2)
    box(lwd=1.5)
    # Plot index
    if (inp$nindex > 0){
        i <- 1
        main <- paste0('Nobs I: ', inp$nobsI[i])
        plot(inp$timeI[[i]], inp$obsI[[i]], typ='l', ylab=paste('Index', i), xlab='Time',
             main=main, xlim=xlim)
        grid()
        plot.col(inp$timeI[[i]], inp$obsI[[i]], pch=i, do.line=FALSE, cex=0.6, add=TRUE,
                 add.legend=FALSE)
        if(inp$nindex>1 & is.null(one.index)){
            for(i in 2:inp$nindex){
                main <- paste0('Nobs I: ', inp$nobsI[i])
                plot(inp$timeI[[i]], inp$obsI[[i]], typ='l', ylab=paste('Index', i),
                     xlab='Time', main=main, xlim=xlim)
                grid()
                plot.col(inp$timeI[[i]], inp$obsI[[i]], pch=i, do.line=FALSE, cex=0.6,
                         add=TRUE, add.legend=FALSE)
            }
        }
        box(lwd=1.5)
    }
    # Plot effort
    if (inp$nobsE){
        main <- paste0('Nobs E: ', inp$nobsE)
        ylab <- 'Effort'
        plot(inp$timeE, inp$obsE, typ='l', ylab=ylab, xlab='Time', main=main, xlim=xlim)
        grid()
        plot.col(inp$timeE, inp$obsE, do.line=FALSE, cex=0.6, add=TRUE, add.legend=FALSE)
        box(lwd=1.5)
    }
    txt.stamp(stamp)
}


