# Stochastic surplus Production model in Continuous-Time (SPiCT)
#    Copyright (C) 2015  Martin Waever Pedersen, mawp@dtu.dk or wpsgodd@gmail.com
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


#' @name calc.osa.resid
#' @title Calculate one-step-ahead residuals.
#' @details In TMB one-step-ahead residuals are calculated by sequentially including one data point at a time while keeping the model parameters fixed at their ML estimates. The calculated residuals are tested for independence, bias, and normality. 
#' @param rep A result report as generated by running fit.spict.
#' @return An updated result report, which contains one-step-ahead residuals stored in $osarC and $osarI.
#' @export
#' @examples
#' data(pol)
#' rep <- fit.spict(pol$albacore)
#' rep <- calc.osa.resid(rep)
#' plotspict.osar(rep)
#' @import TMB
calc.osa.resid <- function(rep){
    doflag <- TRUE
    if('sderr' %in% names(rep)) doflag <- rep$sderr != 1
    if(doflag){
        inp <- rep$inp
        # - Built-in OSAR -
        if(rep$inp$osar.method == 'none'){
            rep$inp$osar.method <- 'oneStepGaussianOffMode' # New default
        }
        if(inp$osar.trace) cat('Number of OSAR steps:', length(rep$inp$osar.subset), '\n')
        osar <- try(oneStepPredict(rep$obj, observation.name = "obssrt", data.term.indicator='keep', method=rep$inp$osar.method, discrete=FALSE, conditional=rep$inp$osar.conditional, subset=rep$inp$osar.subset, trace=inp$osar.trace, parallel=inp$osar.parallel))
        if(class(osar) != 'try-error'){
            osar <- cbind(id=inp$obsidsrt[inp$osar.subset], osar)
            # Store catch residuals separately
            inds <- match(inp$obsidC, osar$id)
            inds <- inds[!is.na(inds)]
            rep$osarC <- osar[inds, ]
            inds2 <- match(osar$id, inp$obsidC)
            inds2 <- inds2[!is.na(inds2)]
            timeC <- inp$timeC[inds2]
            # Store index residuals separately
            rep$osarI <- list()
            timeI <- list()
            for(i in 1:rep$inp$nindex){
                inds <- match(inp$obsidI[[i]], osar$id)
                inds <- inds[!is.na(inds)]
                rep$osarI[[i]] <- osar[inds,]
                inds2 <- match(osar$id, inp$obsidI[[i]])
                inds2 <- inds2[!is.na(inds2)]
                timeI[[i]] <- inp$timeI[[i]][inds2]
            }
            npar <- length(rep$opt$par)
            if(!'stats' %in% names(rep)) rep$stats <- list()
            # Catches
            logCpres <- rep$osarC$residual
            statsCp <- res.stats(logCpres, name='catch')
            for(nm in names(statsCp$stats)) rep$stats[[nm]] <- statsCp$stats[[nm]]
            # Indices
            logIpres <- list()
            logIpshapiro <- list()
            logIpbias <- list()
            statsIp <- list()
            for(i in 1:inp$nindex){
                logIpres[[i]] <- rep$osarI[[i]]$residual
                #logIpres[[i]][1] <- NA # Always omit first residual because it can be difficult to calculate
                statsIp[[i]] <- res.stats(logIpres[[i]], name=paste0('index', i))
                logIpshapiro[[i]] <- statsIp[[i]]$shapiro
                logIpbias[[i]] <- statsIp[[i]]$bias
                nam <- paste0('acfI', i, '.p')
                rep$stats[[nam]] <- statsIp[[i]]$stats$acf.p
                nam <- paste0('shapiroI', i, '.p')
                rep$stats[[nam]] <- logIpshapiro[[i]]$p.value
                nam <- paste0('biasI', i, '.p')
                rep$stats[[nam]] <- logIpbias[[i]]$p.value
            }
            rep$osar <- list(timeC=timeC, logCpres=logCpres, logCpbias=statsCp$bias, logCpshapiro=statsCp$shapiro, timeI=timeI, logIpres=logIpres, logIpshapiro=logIpshapiro, logIpbias=logIpbias)
        } else {
            stop('Could not calculate OSA residuals.\n')
        }
    } else {
        stop('Could not calculate OSA residuals because sdreport() resulted in an error.\n')
    }
    return(rep)
}


#' @name res.stats
#' @title Helper function for calc.osar.resid that calculates residual statistics.
#' @param resid Residuals from either catches or indices.
#' @param name Identifier that will be used in warning messages.
#' @return List containing residual statistics in 'stats', shapiro output in 'shapiro', and bias output in 'bias'.
#' @export
res.stats <- function(resid, name=''){
    nna <- sum(is.na(resid))
    if(nna > 0) warning(nna, ' NAs found in ', name, ' residuals')
    nnotna <- sum(!is.na(resid))
    stats <- list()
    if(nnotna > 2){
        shapiro <- shapiro.test(resid) # Test for normality of residuals
        bias <- t.test(resid) # Test for bias of residuals
        stats$acf.p <- min(acf.signf(resid, lag.max=4, return.p=TRUE))
    } else {
        warning('Warning: only ', nnotna, ' non-NAs found in ', name, ' residuals. Not calculating residual statistics')
        bias <- list(statistic=NA, p.value=NA, method=NA, data.name=NA)
        shapiro <- list(statistic=NA, p.value=NA, method=NA, data.name=NA)
    }
    stats$shapiro.p <- shapiro$p.value
    stats$bias.p <- bias$p.value
    if(is.null(stats$acf.p)) stats$acf.p <- NA
    return(list(shapiro=shapiro, bias=bias, stats=stats))
}
