library(TMB)
library(numDeriv)

makeTemplate <- function(expr) {
    txt <- c(
        "// Autogenerated by check_derivatives.R",
        "#include <TMB.hpp>",
        "template<class Type>",
        "Type objective_function<Type>::operator() ()",
        "{",
        "  PARAMETER_VECTOR(x_);",
        "  vector<Type> x(x_.size()+1);",
        "  x << 0, x_;",
        paste("  return",deparse(expr),";"),
        "}"
    )
    writeLines(txt,"autogenerated.cpp")
    compile("autogenerated.cpp")
    dyn.load(dynlib("autogenerated"))
}

.resultList <- list()
checkDerivatives <- function(exprR,
                             exprTMB=NULL, ## Auto-generated
                             grid, fcpp=FALSE) {
    exprR <- substitute(exprR)
    if(!missing(exprTMB)) exprTMB <- substitute(exprTMB)
    else exprTMB <- exprR
    makeTemplate(exprTMB)
    obj <- MakeADFun(data=list(),
                     parameters=list(x_=unlist(grid[1,])),
                     DLL="autogenerated", silent=TRUE)
    Type <- function(x)x
    f <- function(x)eval(exprR)
    if(fcpp) f <- obj$fn
    r <- function(x, y) {
        stopifnot(identical(dim(x), dim(y)))
        ans <- 2. * abs(x - y) / (abs(x) + abs(y))
        if(is.matrix(ans)) ans <- base::apply(ans, 2, max)
        ans
    }
    apply <- function(...){
        ans <- base::apply(...)
        if(is.vector(ans))
            ans <- t(ans)
        ans
    }
    ## value
    v1 <- apply(grid, 1, f)
    v2 <- apply(grid, 1, obj$fn)
    rv <- r(v1, v2)
    ## gradient
    g1 <- apply(grid, 1, grad, func=f)
    g2 <- apply(grid, 1, obj$gr)
    rg <- r(g1, g2)
    ## hessian
    h1 <- apply(grid, 1, hessian, func=f)
    h2 <- apply(grid, 1, obj$he)
    rh <- r(h1, h2)
    ans <- as.list(environment())
    class(ans) <- "checkDerivatives"
    .GlobalEnv$.resultList[[length(.resultList) + 1]] <- ans
    ans
}

print.checkDerivatives <- function(x,...) {
    local({
        cat("=================================================================\n")
        cat(exprR[[1]],"\n")
        mycat <- function(what="gradient", g1, g2, rg) {
            What <- what
            substring(What,1,1) <- toupper(substring(what,1,1))
            cat("\n")
            cat(What, "relative error:\n")
            print(summary(rg))
            cat("\nWorst", what, "grid-point (numderiv/ad):\n")
            cat("x  =",unlist(grid[which.max(rg),]),"\n")
            cat("ND =",g1[,which.max(rg)],"\n")
            cat("AD =",g2[,which.max(rg)],"\n")
        }
        mycat("value", v1, v2, rv)
        mycat("gradient", g1, g2, rg)
        mycat("hessian", h1, h2, rh)
        ##cat("=================================================================\n")
    }, x)
}


## Common grid for Bessel functions
grid <- expand.grid(1:100, 1:100) / 10

checkDerivatives(
    besselK(x[1], x[2]),
    grid = grid
)

checkDerivatives(
    besselI(x[1], x[2]),
    grid = grid
)

checkDerivatives(
    besselY(x[1], x[2]),
    grid = grid
)

## Grid for qbeta and pbeta
grid <- expand.grid(1:9 + .05, 1:9 * 2 - .1, 1:9 * 2 + .1) / 10
checkDerivatives(
    qbeta(x[1], x[2], x[3]),
    grid = grid
)
checkDerivatives(
    pbeta(x[1], x[2], x[3]),
    grid = grid
)

checkDerivatives(
    pbeta(x[1], x[2], x[3]),
    grid = grid,
    fcpp=TRUE
)

library(tweedie)
grid <- expand.grid(p=c(1.2,1.5,1.8), mu=(1:3)*5, phi=1:3)
checkDerivatives(
    dtweedie(5, x[1], x[2], x[3]),
    grid = grid,
    dtweedie(Type(5), x[2], x[3], x[1], false)
)

grid <- expand.grid(x=(1:100)/10)
checkDerivatives(
    besselI(x[1], Type(1.0)),
    grid = grid
)

##invisible(lapply(.resultList,print))
